# WhatsApp CRM API

A complete WhatsApp CRM API system with QR code authentication, message management, contact management, and message templates.

## Features

✅ **QR Code Authentication** - Secure WhatsApp Web QR login
✅ **Message Management** - Send and receive messages through API
✅ **Contact Management** - Manage WhatsApp contacts
✅ **Message Templates** - Create reusable message templates with variable support
✅ **API Key System** - Each session gets a unique API key
✅ **SQLite Database** - Persistent storage of sessions, messages, and contacts
✅ **Real-time Updates** - Receive messages as they arrive
✅ **CRM Ready** - Build your CRM on top of this API

## Installation

### Prerequisites
- Node.js 14+
- npm or yarn

### Setup

1. **Clone/Navigate to the project**
```bash
cd whatsappcrm
```

2. **Install dependencies**
```bash
npm install
```

3. **Configure environment variables**

Edit `.env` file:
```env
PORT=3000
NODE_ENV=development
API_KEY=your_secret_api_key_change_this
DATABASE_PATH=./data/whatsapp.db
SESSION_PATH=./sessions
LOG_PATH=./logs
```

4. **Start the server**
```bash
npm start
```

Or for development with auto-reload:
```bash
npm run dev
```

The server will start at `http://localhost:3000`

## Quick Start

### 1. Initialize a Session

```bash
curl -X POST http://localhost:3000/api/auth/init
```

Response:
```json
{
  "success": true,
  "sessionId": "uuid-here",
  "apiKey": "long-api-key-here",
  "status": "pending",
  "message": "Session initialized. Wait for QR code and scan it with your WhatsApp.",
  "qrCodeUrl": "/api/auth/uuid-here/qr"
}
```

### 2. Get QR Code

```bash
curl http://localhost:3000/api/auth/YOUR_SESSION_ID/qr
```

This returns a QR code as data URL. Open it in your browser and scan with WhatsApp.

### 3. Check Authentication Status

```bash
curl http://localhost:3000/api/auth/YOUR_SESSION_ID/status \
  -H "X-API-Key: YOUR_API_KEY"
```

### 4. Send a Message

```bash
curl -X POST http://localhost:3000/api/messages/YOUR_SESSION_ID/send \
  -H "X-API-Key: YOUR_API_KEY" \
  -H "Content-Type: application/json" \
  -d '{
    "phone": "919876543210",
    "message": "Hello from WhatsApp CRM API!"
  }'
```

### 5. Get Contacts

```bash
curl http://localhost:3000/api/contacts/YOUR_SESSION_ID \
  -H "X-API-Key: YOUR_API_KEY"
```

## API Endpoints

### Authentication

| Method | Endpoint | Description |
|--------|----------|-------------|
| POST | `/api/auth/init` | Initialize a new session with QR code |
| GET | `/api/auth/:sessionId/qr` | Get QR code for scanning |
| GET | `/api/auth/:sessionId/status` | Check session status |

### Messages

| Method | Endpoint | Description |
|--------|----------|-------------|
| POST | `/api/messages/:sessionId/send` | Send a message |
| GET | `/api/messages/:sessionId/list` | Get all messages |
| GET | `/api/messages/:sessionId/conversation/:phone` | Get conversation with a contact |

### Contacts

| Method | Endpoint | Description |
|--------|----------|-------------|
| POST | `/api/contacts/:sessionId` | Add a new contact |
| GET | `/api/contacts/:sessionId` | Get all contacts |
| GET | `/api/contacts/:sessionId/:phone` | Get specific contact |
| PUT | `/api/contacts/:sessionId/:id` | Update a contact |
| DELETE | `/api/contacts/:sessionId/:id` | Delete a contact |

### Templates

| Method | Endpoint | Description |
|--------|----------|-------------|
| POST | `/api/templates/:sessionId` | Create a template |
| GET | `/api/templates/:sessionId` | Get all templates |
| GET | `/api/templates/:sessionId/:id` | Get specific template |
| PUT | `/api/templates/:sessionId/:id` | Update a template |
| POST | `/api/templates/:sessionId/:id/render` | Render template with variables |
| DELETE | `/api/templates/:sessionId/:id` | Delete a template |

## Authentication

All endpoints (except `/auth/init` and `/auth/:sessionId/qr`) require authentication using API Key:

```
Header: X-API-Key: your_api_key_here
```

## Example Use Cases

### Create a Template

```bash
curl -X POST http://localhost:3000/api/templates/YOUR_SESSION_ID \
  -H "X-API-Key: YOUR_API_KEY" \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Welcome Message",
    "content": "Hello {{name}}, welcome to our service!",
    "variables": ["name"]
  }'
```

### Render Template

```bash
curl -X POST http://localhost:3000/api/templates/YOUR_SESSION_ID/TEMPLATE_ID/render \
  -H "X-API-Key: YOUR_API_KEY" \
  -H "Content-Type: application/json" \
  -d '{
    "variables": {
      "name": "John Doe"
    }
  }'
```

### Send Using Template

```bash
# First render the template
# Then send the rendered message
curl -X POST http://localhost:3000/api/messages/YOUR_SESSION_ID/send \
  -H "X-API-Key: YOUR_API_KEY" \
  -H "Content-Type: application/json" \
  -d '{
    "phone": "919876543210",
    "message": "Hello John Doe, welcome to our service!"
  }'
```

## Database Schema

### Sessions Table
- `id` - Unique session identifier
- `phone_number` - WhatsApp phone number (after authentication)
- `api_key` - Unique API key for this session
- `qr_code` - QR code data URL
- `status` - Session status (pending, authenticated, disconnected)
- `created_at` - Creation timestamp
- `updated_at` - Last update timestamp

### Messages Table
- `id` - Message ID
- `session_id` - Associated session
- `from_phone` - Sender's phone number
- `to_phone` - Recipient's phone number
- `message_text` - Message content
- `message_type` - Type (text, media, etc.)
- `direction` - incoming or outgoing
- `timestamp` - Message timestamp
- `read_status` - Whether message was read
- `media_url` - URL if message contains media

### Contacts Table
- `id` - Contact ID
- `session_id` - Associated session
- `phone` - Contact's phone number
- `name` - Contact's name
- `avatar` - Contact's avatar URL
- `created_at` - When contact was added

### Templates Table
- `id` - Template ID
- `session_id` - Associated session
- `name` - Template name
- `content` - Template content with {{variables}}
- `variables` - JSON array of variable names
- `created_at` - Creation timestamp

## Error Handling

All errors are returned with appropriate HTTP status codes:

```json
{
  "error": "Error Type",
  "details": "Detailed error message"
}
```

## Security Notes

1. ⚠️ Change the default API_KEY in `.env`
2. ⚠️ Use HTTPS in production
3. ⚠️ Keep API keys secure
4. ⚠️ Don't expose session IDs publicly
5. ⚠️ Implement rate limiting in production

## Project Structure

```
whatsappcrm/
├── src/
│   ├── index.js              # Main server file
│   ├── database.js           # Database operations
│   ├── auth.js               # Authentication logic
│   ├── whatsapp.js           # WhatsApp Web integration
│   └── routes/
│       ├── auth.js           # Auth endpoints
│       ├── messages.js       # Message endpoints
│       ├── contacts.js       # Contact endpoints
│       └── templates.js      # Template endpoints
├── data/                     # SQLite database
├── sessions/                 # WhatsApp sessions
├── logs/                     # Application logs
├── .env                      # Environment variables
├── package.json              # Dependencies
└── README.md                 # This file
```

## Development

For development with hot reload:

```bash
npm install --save-dev nodemon
npm run dev
```

## Production Deployment

1. Set `NODE_ENV=production` in `.env`
2. Use a process manager like PM2:
```bash
npm install -g pm2
pm2 start src/index.js --name "whatsapp-crm-api"
```
3. Set up reverse proxy (nginx/Apache)
4. Use HTTPS certificates
5. Implement rate limiting and authentication

## Troubleshooting

### QR Code Not Showing
- Wait 10-15 seconds after initialization
- Check that Node.js process is running
- Check console for any errors

### Messages Not Receiving
- Ensure WhatsApp is not logged in elsewhere
- Check that session is authenticated
- Verify phone number format (include country code)

### API Key Errors
- Verify API key is passed in `X-API-Key` header
- Check that session is authenticated first
- Ensure session ID matches the API key

## License

ISC

## Support

For issues and questions, please create an issue in the repository.
