# API Testing Guide

This document contains ready-to-use test examples for the WhatsApp CRM API.

## Prerequisites

- Running WhatsApp CRM API server
- cURL or Postman installed
- Valid WhatsApp account

## Setup Test Variables

```bash
# Set these variables for easy copying/pasting
SESSION_ID="your_session_id_here"
API_KEY="your_api_key_here"
PHONE="919876543210"
API_BASE="http://localhost:3000/api"
```

---

## Test Sequences

### Sequence 1: Complete Authentication Flow

#### 1.1 Initialize Session
```bash
curl -X POST http://localhost:3000/api/auth/init \
  -H "Content-Type: application/json"
```

**Expected Response:**
```json
{
  "success": true,
  "sessionId": "550e8400-e29b-41d4-a716-446655440000",
  "apiKey": "abc123def456abc123def456...",
  "status": "pending",
  "message": "Session initialized. Wait for QR code and scan it with your WhatsApp.",
  "qrCodeUrl": "/api/auth/550e8400-e29b-41d4-a716-446655440000/qr"
}
```

Save the `sessionId` and `apiKey` for next requests.

#### 1.2 Get QR Code
```bash
curl http://localhost:3000/api/auth/550e8400-e29b-41d4-a716-446655440000/qr
```

**Expected Response (Before Scanning):**
```json
{
  "status": "pending",
  "message": "QR code not yet generated. Please wait..."
}
```

**Expected Response (After Generation):**
```json
{
  "status": "pending",
  "qr_code": "data:image/png;base64,iVBORw0KGgo..."
}
```

> ⚠️ Open the QR code in a browser and scan with WhatsApp mobile app

#### 1.3 Check Status (Repeat until authenticated)
```bash
curl http://localhost:3000/api/auth/550e8400-e29b-41d4-a716-446655440000/status \
  -H "X-API-Key: abc123def456abc123def456..."
```

**Expected Response (Pending):**
```json
{
  "sessionId": "550e8400-e29b-41d4-a716-446655440000",
  "status": "pending",
  "phone_number": null,
  "api_key": "abc123def456abc123def456...",
  "created_at": "2024-01-30T10:00:00Z"
}
```

**Expected Response (Authenticated):**
```json
{
  "sessionId": "550e8400-e29b-41d4-a716-446655440000",
  "status": "authenticated",
  "phone_number": "919876543210",
  "api_key": "abc123def456abc123def456...",
  "created_at": "2024-01-30T10:00:00Z"
}
```

---

### Sequence 2: Message Operations

#### 2.1 Send Message
```bash
curl -X POST http://localhost:3000/api/messages/550e8400-e29b-41d4-a716-446655440000/send \
  -H "X-API-Key: abc123def456abc123def456..." \
  -H "Content-Type: application/json" \
  -d '{
    "phone": "919876543210",
    "message": "Hello from WhatsApp CRM API!"
  }'
```

**Expected Response:**
```json
{
  "success": true,
  "messageId": "true_919876543210@c.us_E1234567890ABCDEF",
  "timestamp": "2024-01-30T10:05:00.000Z",
  "message": "Message sent successfully"
}
```

#### 2.2 Get All Messages
```bash
curl "http://localhost:3000/api/messages/550e8400-e29b-41d4-a716-446655440000/list?limit=10&offset=0" \
  -H "X-API-Key: abc123def456abc123def456..."
```

**Expected Response:**
```json
{
  "success": true,
  "total": 5,
  "messages": [
    {
      "id": "true_919876543210@c.us_E1234567890ABCDEF",
      "session_id": "550e8400-e29b-41d4-a716-446655440000",
      "from_phone": "919876543210",
      "to_phone": "919876543211@c.us",
      "message_text": "Hello from WhatsApp CRM API!",
      "message_type": "text",
      "direction": "outgoing",
      "timestamp": "2024-01-30T10:05:00Z",
      "read_status": 1
    }
  ]
}
```

#### 2.3 Get Conversation with Specific Contact
```bash
curl "http://localhost:3000/api/messages/550e8400-e29b-41d4-a716-446655440000/conversation/919876543210?limit=20" \
  -H "X-API-Key: abc123def456abc123def456..."
```

**Expected Response:**
```json
{
  "success": true,
  "phone": "919876543210",
  "total": 3,
  "messages": [
    {
      "id": "true_919876543210@c.us_E1234567890ABCDEF",
      "session_id": "550e8400-e29b-41d4-a716-446655440000",
      "from_phone": "919876543211",
      "to_phone": "919876543210",
      "message_text": "Thanks for the message!",
      "message_type": "text",
      "direction": "incoming",
      "timestamp": "2024-01-30T10:06:00Z",
      "read_status": 0
    }
  ]
}
```

---

### Sequence 3: Contact Management

#### 3.1 Add Contact
```bash
curl -X POST http://localhost:3000/api/contacts/550e8400-e29b-41d4-a716-446655440000 \
  -H "X-API-Key: abc123def456abc123def456..." \
  -H "Content-Type: application/json" \
  -d '{
    "phone": "919876543210",
    "name": "John Doe"
  }'
```

**Expected Response:**
```json
{
  "success": true,
  "contactId": "550e8400-e29b-41d4-a716-446655440001",
  "phone": "919876543210",
  "name": "John Doe"
}
```

#### 3.2 Get All Contacts
```bash
curl http://localhost:3000/api/contacts/550e8400-e29b-41d4-a716-446655440000 \
  -H "X-API-Key: abc123def456abc123def456..."
```

**Expected Response:**
```json
{
  "success": true,
  "total": 2,
  "contacts": [
    {
      "id": "550e8400-e29b-41d4-a716-446655440001",
      "session_id": "550e8400-e29b-41d4-a716-446655440000",
      "phone": "919876543210",
      "name": "John Doe",
      "avatar": null,
      "created_at": "2024-01-30T10:00:00Z"
    }
  ]
}
```

#### 3.3 Get Specific Contact
```bash
curl http://localhost:3000/api/contacts/550e8400-e29b-41d4-a716-446655440000/919876543210 \
  -H "X-API-Key: abc123def456abc123def456..."
```

**Expected Response:**
```json
{
  "success": true,
  "contact": {
    "id": "550e8400-e29b-41d4-a716-446655440001",
    "session_id": "550e8400-e29b-41d4-a716-446655440000",
    "phone": "919876543210",
    "name": "John Doe",
    "avatar": null,
    "created_at": "2024-01-30T10:00:00Z"
  }
}
```

#### 3.4 Update Contact
```bash
curl -X PUT http://localhost:3000/api/contacts/550e8400-e29b-41d4-a716-446655440000/550e8400-e29b-41d4-a716-446655440001 \
  -H "X-API-Key: abc123def456abc123def456..." \
  -H "Content-Type: application/json" \
  -d '{
    "name": "John Smith",
    "phone": "919876543210"
  }'
```

**Expected Response:**
```json
{
  "success": true,
  "message": "Contact updated successfully"
}
```

#### 3.5 Delete Contact
```bash
curl -X DELETE http://localhost:3000/api/contacts/550e8400-e29b-41d4-a716-446655440000/550e8400-e29b-41d4-a716-446655440001 \
  -H "X-API-Key: abc123def456abc123def456..."
```

**Expected Response:**
```json
{
  "success": true,
  "message": "Contact deleted successfully"
}
```

---

### Sequence 4: Template Management

#### 4.1 Create Template
```bash
curl -X POST http://localhost:3000/api/templates/550e8400-e29b-41d4-a716-446655440000 \
  -H "X-API-Key: abc123def456abc123def456..." \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Welcome Message",
    "content": "Hello {{name}}, welcome to {{company}}!",
    "variables": ["name", "company"]
  }'
```

**Expected Response:**
```json
{
  "success": true,
  "templateId": "550e8400-e29b-41d4-a716-446655440002",
  "name": "Welcome Message",
  "content": "Hello {{name}}, welcome to {{company}}!"
}
```

#### 4.2 Get All Templates
```bash
curl http://localhost:3000/api/templates/550e8400-e29b-41d4-a716-446655440000 \
  -H "X-API-Key: abc123def456abc123def456..."
```

**Expected Response:**
```json
{
  "success": true,
  "total": 1,
  "templates": [
    {
      "id": "550e8400-e29b-41d4-a716-446655440002",
      "session_id": "550e8400-e29b-41d4-a716-446655440000",
      "name": "Welcome Message",
      "content": "Hello {{name}}, welcome to {{company}}!",
      "variables": ["name", "company"],
      "created_at": "2024-01-30T10:10:00Z"
    }
  ]
}
```

#### 4.3 Render Template
```bash
curl -X POST http://localhost:3000/api/templates/550e8400-e29b-41d4-a716-446655440000/550e8400-e29b-41d4-a716-446655440002/render \
  -H "X-API-Key: abc123def456abc123def456..." \
  -H "Content-Type: application/json" \
  -d '{
    "variables": {
      "name": "John Doe",
      "company": "ACME Corp"
    }
  }'
```

**Expected Response:**
```json
{
  "success": true,
  "rendered": "Hello John Doe, welcome to ACME Corp!"
}
```

#### 4.4 Send Message Using Rendered Template
```bash
curl -X POST http://localhost:3000/api/messages/550e8400-e29b-41d4-a716-446655440000/send \
  -H "X-API-Key: abc123def456abc123def456..." \
  -H "Content-Type: application/json" \
  -d '{
    "phone": "919876543210",
    "message": "Hello John Doe, welcome to ACME Corp!"
  }'
```

#### 4.5 Update Template
```bash
curl -X PUT http://localhost:3000/api/templates/550e8400-e29b-41d4-a716-446655440000/550e8400-e29b-41d4-a716-446655440002 \
  -H "X-API-Key: abc123def456abc123def456..." \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Welcome Message Updated",
    "content": "Hey {{name}}, great to have you at {{company}}!",
    "variables": ["name", "company"]
  }'
```

**Expected Response:**
```json
{
  "success": true,
  "message": "Template updated successfully"
}
```

#### 4.6 Delete Template
```bash
curl -X DELETE http://localhost:3000/api/templates/550e8400-e29b-41d4-a716-446655440000/550e8400-e29b-41d4-a716-446655440002 \
  -H "X-API-Key: abc123def456abc123def456..."
```

**Expected Response:**
```json
{
  "success": true,
  "message": "Template deleted successfully"
}
```

---

## Postman Collection

### Import into Postman

1. Create a new collection called "WhatsApp CRM API"
2. Create a variable: `base_url` = `http://localhost:3000`
3. Create a variable: `api_key` = (your API key)
4. Create a variable: `session_id` = (your session ID)

### Use in Requests

Replace hardcoded values with:
- `{{base_url}}`
- `{{api_key}}`
- `{{session_id}}`

---

## Common Error Scenarios

### Error: 401 Unauthorized
**Cause:** Missing or invalid API key
**Solution:** 
1. Check header name is exactly `X-API-Key`
2. Verify API key is correct
3. Ensure session is authenticated

### Error: 404 Not Found
**Cause:** Invalid endpoint or resource not found
**Solution:**
1. Check endpoint URL is correct
2. Verify session/contact/template IDs exist
3. Check documentation for correct format

### Error: Session not authenticated
**Cause:** Trying to use API before scanning QR code
**Solution:**
1. Check status with `/auth/:sessionId/status`
2. Scan QR code with WhatsApp
3. Wait for status to change to "authenticated"

### Error: Invalid phone number
**Cause:** Phone number format is incorrect
**Solution:**
1. Use international format (e.g., 919876543210)
2. Don't include + or spaces
3. Ensure it's a valid WhatsApp number

---

## Performance Testing

### Send Multiple Messages
```bash
for i in {1..10}; do
  curl -X POST http://localhost:3000/api/messages/550e8400-e29b-41d4-a716-446655440000/send \
    -H "X-API-Key: abc123def456abc123def456..." \
    -H "Content-Type: application/json" \
    -d "{\"phone\": \"919876543210\", \"message\": \"Message $i\"}"
  sleep 1
done
```

### Bulk Contact Creation
```bash
for i in {1..100}; do
  curl -X POST http://localhost:3000/api/contacts/550e8400-e29b-41d4-a716-446655440000 \
    -H "X-API-Key: abc123def456abc123def456..." \
    -H "Content-Type: application/json" \
    -d "{\"phone\": \"9198765432${i:0:2}\", \"name\": \"Contact $i\"}"
done
```

---

## Notes

- Replace `550e8400-e29b-41d4-a716-446655440000` with your actual session ID
- Replace `abc123def456abc123def456...` with your actual API key
- Replace `919876543210` with actual phone numbers (international format)
- For batch operations, add delays to avoid rate limiting
