/**
 * WhatsApp CRM API Client
 * Simple client library for interacting with the WhatsApp CRM API
 */

class WhatsAppCRMClient {
  constructor(baseUrl = 'http://localhost:3000', apiKey = null, sessionId = null) {
    this.baseUrl = baseUrl;
    this.apiKey = apiKey;
    this.sessionId = sessionId;
  }

  async request(method, endpoint, data = null) {
    const url = `${this.baseUrl}${endpoint}`;
    const options = {
      method,
      headers: {
        'Content-Type': 'application/json'
      }
    };

    if (this.apiKey && this.sessionId) {
      options.headers['X-API-Key'] = this.apiKey;
    }

    if (data) {
      options.body = JSON.stringify(data);
    }

    try {
      const response = await fetch(url, options);
      return await response.json();
    } catch (error) {
      console.error('Request failed:', error);
      throw error;
    }
  }

  // Authentication
  async initSession() {
    const response = await this.request('POST', '/api/auth/init');
    this.sessionId = response.sessionId;
    this.apiKey = response.apiKey;
    return response;
  }

  async getQRCode() {
    return this.request('GET', `/api/auth/${this.sessionId}/qr`);
  }

  async getSessionStatus() {
    return this.request('GET', `/api/auth/${this.sessionId}/status`);
  }

  // Messages
  async sendMessage(phone, message, mediaPath = null) {
    return this.request('POST', `/api/messages/${this.sessionId}/send`, {
      phone,
      message,
      mediaPath
    });
  }

  async getMessages(limit = 50, offset = 0) {
    return this.request('GET', `/api/messages/${this.sessionId}/list?limit=${limit}&offset=${offset}`);
  }

  async getConversation(phone, limit = 50, offset = 0) {
    return this.request('GET', 
      `/api/messages/${this.sessionId}/conversation/${phone}?limit=${limit}&offset=${offset}`
    );
  }

  // Contacts
  async addContact(phone, name) {
    return this.request('POST', `/api/contacts/${this.sessionId}`, { phone, name });
  }

  async getContacts() {
    return this.request('GET', `/api/contacts/${this.sessionId}`);
  }

  async getContact(phone) {
    return this.request('GET', `/api/contacts/${this.sessionId}/${phone}`);
  }

  async updateContact(id, name, phone) {
    return this.request('PUT', `/api/contacts/${this.sessionId}/${id}`, { name, phone });
  }

  async deleteContact(id) {
    return this.request('DELETE', `/api/contacts/${this.sessionId}/${id}`);
  }

  // Templates
  async createTemplate(name, content, variables = []) {
    return this.request('POST', `/api/templates/${this.sessionId}`, {
      name,
      content,
      variables
    });
  }

  async getTemplates() {
    return this.request('GET', `/api/templates/${this.sessionId}`);
  }

  async getTemplate(id) {
    return this.request('GET', `/api/templates/${this.sessionId}/${id}`);
  }

  async updateTemplate(id, name, content, variables = []) {
    return this.request('PUT', `/api/templates/${this.sessionId}/${id}`, {
      name,
      content,
      variables
    });
  }

  async renderTemplate(id, variables) {
    return this.request('POST', `/api/templates/${this.sessionId}/${id}/render`, {
      variables
    });
  }

  async deleteTemplate(id) {
    return this.request('DELETE', `/api/templates/${this.sessionId}/${id}`);
  }

  // Utility
  async waitForAuthentication(maxWaitTime = 60000, checkInterval = 2000) {
    const startTime = Date.now();
    
    while (Date.now() - startTime < maxWaitTime) {
      try {
        const status = await this.getSessionStatus();
        if (status.status === 'authenticated') {
          return status;
        }
      } catch (error) {
        // Ignore errors during polling
      }
      
      await new Promise(resolve => setTimeout(resolve, checkInterval));
    }
    
    throw new Error('Authentication timeout - QR code was not scanned within the time limit');
  }
}

module.exports = WhatsAppCRMClient;
