require('dotenv').config();
const express = require('express');
const bodyParser = require('body-parser');
const cors = require('cors');
const path = require('path');
const fs = require('fs');

// Import modules
const { initializeDatabase } = require('./database');
const authRoutes = require('./routes/auth');
const messagesRoutes = require('./routes/messages');
const contactsRoutes = require('./routes/contacts');
const templatesRoutes = require('./routes/templates');
const webhooksRoutes = require('./routes/webhooks');

const app = express();
const PORT = process.env.PORT || 3000;

// Ensure necessary directories exist
const dirs = [
  process.env.SESSION_PATH || './sessions',
  process.env.DATABASE_PATH || './data',
  process.env.LOG_PATH || './logs'
];

dirs.forEach(dir => {
  if (!fs.existsSync(dir)) {
    fs.mkdirSync(dir, { recursive: true });
  }
});

// Middleware
app.use(cors());
app.use(bodyParser.json({ limit: '50mb' }));
app.use(bodyParser.urlencoded({ limit: '50mb', extended: true }));

// Routes
app.use('/api/auth', authRoutes);
app.use('/api/messages', messagesRoutes);
app.use('/api/contacts', contactsRoutes);
app.use('/api/templates', templatesRoutes);
app.use('/api/webhooks', webhooksRoutes);

// Health check endpoint
app.get('/health', (req, res) => {
  res.json({
    status: 'ok',
    timestamp: new Date(),
    message: 'WhatsApp CRM API is running'
  });
});

// API Documentation endpoint
app.get('/api/docs', (req, res) => {
  res.json({
    name: 'WhatsApp CRM API',
    version: '1.0.0',
    description: 'Complete WhatsApp CRM API with QR Authentication and Webhooks',
    baseUrl: `http://localhost:${PORT}/api`,
    endpoints: {
      authentication: {
        'POST /auth/init': 'Initialize a new WhatsApp session with QR code',
        'GET /auth/:sessionId/qr': 'Get the QR code image',
        'GET /auth/:sessionId/status': 'Check session authentication status'
      },
      messages: {
        'POST /messages/:sessionId/send': 'Send a message to a WhatsApp contact',
        'GET /messages/:sessionId/list': 'Get all messages',
        'GET /messages/:sessionId/conversation/:phone': 'Get conversation with a specific contact'
      },
      contacts: {
        'POST /contacts/:sessionId': 'Add a new contact',
        'GET /contacts/:sessionId': 'Get all contacts',
        'GET /contacts/:sessionId/:phone': 'Get a specific contact',
        'PUT /contacts/:sessionId/:id': 'Update a contact',
        'DELETE /contacts/:sessionId/:id': 'Delete a contact'
      },
      templates: {
        'POST /templates/:sessionId': 'Create a message template',
        'GET /templates/:sessionId': 'Get all templates',
        'GET /templates/:sessionId/:id': 'Get a specific template',
        'PUT /templates/:sessionId/:id': 'Update a template',
        'POST /templates/:sessionId/:id/render': 'Render a template with variables',
        'DELETE /templates/:sessionId/:id': 'Delete a template'
      },
      webhooks: {
        'POST /webhooks/:sessionId': 'Register a webhook for events',
        'GET /webhooks/:sessionId': 'Get all webhooks',
        'GET /webhooks/:sessionId/:webhookId': 'Get specific webhook',
        'PUT /webhooks/:sessionId/:webhookId': 'Update webhook',
        'DELETE /webhooks/:sessionId/:webhookId': 'Delete webhook',
        'POST /webhooks/:sessionId/:webhookId/test': 'Test webhook with sample payload',
        'GET /webhooks/:sessionId/:webhookId/logs': 'Get webhook delivery logs'
      }
    },
    webhookEvents: [
      'message.received - When a message is received',
      'message.sent - When a message is sent',
      'contact.added - When a contact is added',
      'contact.updated - When a contact is updated',
      'contact.deleted - When a contact is deleted',
      'session.authenticated - When session is authenticated',
      'session.disconnected - When session is disconnected'
    ],
    authentication: {
      type: 'API Key',
      header: 'X-API-Key',
      example: 'X-API-Key: your_api_key_here'
    }
  });
});

// Error handling middleware
app.use((err, req, res, next) => {
  console.error('Error:', err);
  res.status(500).json({
    error: 'Internal Server Error',
    message: err.message
  });
});

// 404 handler
app.use((req, res) => {
  res.status(404).json({
    error: 'Not Found',
    message: 'The requested endpoint does not exist'
  });
});

// Initialize database and start server
async function startServer() {
  try {
    const dbPath = process.env.DATABASE_PATH || './data/whatsapp.db';
    await initializeDatabase(dbPath);
    console.log('Database initialized successfully');

    app.listen(PORT, () => {
      console.log(`
╔═══════════════════════════════════════════════════════╗
║        WhatsApp CRM API Server Started                ║
╠═══════════════════════════════════════════════════════╣
║  Server: http://localhost:${PORT}
║  Health Check: http://localhost:${PORT}/health
║  Documentation: http://localhost:${PORT}/api/docs
╚═══════════════════════════════════════════════════════╝
      `);
    });
  } catch (err) {
    console.error('Failed to start server:', err);
    process.exit(1);
  }
}

startServer();
