const express = require('express');
const router = express.Router();
const { createSession, validateApiKey, getSessionByApiKey } = require('../auth');
const { createWhatsAppSession, getSessionStatus } = require('../whatsapp');

// Middleware to validate API key
async function validateApiKeyMiddleware(req, res, next) {
  const apiKey = req.headers['x-api-key'];
  const sessionId = req.params.sessionId || req.body.sessionId;

  if (!apiKey || !sessionId) {
    return res.status(401).json({ error: 'Missing API key or session ID' });
  }

  const isValid = await validateApiKey(apiKey, sessionId);
  if (!isValid) {
    return res.status(401).json({ error: 'Invalid API key or session not authenticated' });
  }

  req.sessionId = sessionId;
  req.apiKey = apiKey;
  next();
}

// POST /api/auth/init - Initialize a new session
router.post('/init', async (req, res) => {
  try {
    const session = await createSession();
    const whatsappSession = await createWhatsAppSession(session.sessionId);

    res.json({
      success: true,
      sessionId: session.sessionId,
      apiKey: session.apiKey,
      status: 'pending',
      message: 'Session initialized. Wait for QR code and scan it with your WhatsApp.',
      qrCodeUrl: `/api/auth/${session.sessionId}/qr`
    });
  } catch (err) {
    console.error('Error initializing session:', err);
    res.status(500).json({ error: 'Failed to initialize session', details: err.message });
  }
});

// GET /api/auth/:sessionId/qr - Get QR code
router.get('/:sessionId/qr', async (req, res) => {
  try {
    const session = await getSessionStatus(req.params.sessionId);
    
    if (!session) {
      return res.status(404).json({ error: 'Session not found' });
    }

    if (!session.qr_code) {
      return res.status(202).json({
        status: session.status,
        message: 'QR code not yet generated. Please wait...'
      });
    }

    res.json({
      status: session.status,
      qr_code: session.qr_code
    });
  } catch (err) {
    console.error('Error getting QR code:', err);
    res.status(500).json({ error: 'Failed to get QR code', details: err.message });
  }
});

// GET /api/auth/:sessionId/status - Check session status
router.get('/:sessionId/status', async (req, res) => {
  try {
    const session = await getSessionStatus(req.params.sessionId);
    
    if (!session) {
      return res.status(404).json({ error: 'Session not found' });
    }

    res.json({
      sessionId: session.id,
      status: session.status,
      phone_number: session.phone_number,
      api_key: session.api_key,
      created_at: session.created_at
    });
  } catch (err) {
    console.error('Error checking status:', err);
    res.status(500).json({ error: 'Failed to check status', details: err.message });
  }
});

module.exports = router;
