const express = require('express');
const router = express.Router();
const { sendMessage, getSessionMessages, getSessionContacts, logoutSession } = require('../whatsapp');
const { validateApiKey } = require('../auth');

// Middleware to validate API key
async function validateApiKeyMiddleware(req, res, next) {
  const apiKey = req.headers['x-api-key'];
  const sessionId = req.params.sessionId;

  if (!apiKey || !sessionId) {
    return res.status(401).json({ error: 'Missing API key or session ID' });
  }

  const isValid = await validateApiKey(apiKey, sessionId);
  if (!isValid) {
    return res.status(401).json({ error: 'Invalid API key or session not authenticated' });
  }

  req.sessionId = sessionId;
  req.apiKey = apiKey;
  next();
}

// Apply middleware to all routes
router.use('/:sessionId', validateApiKeyMiddleware);

// POST /api/messages/:sessionId/send - Send a message
router.post('/:sessionId/send', async (req, res) => {
  try {
    const { phone, message, mediaPath } = req.body;

    if (!phone || !message) {
      return res.status(400).json({ error: 'Phone and message are required' });
    }

    const result = await sendMessage(req.sessionId, phone, message, mediaPath);

    res.json({
      success: true,
      messageId: result.messageId,
      timestamp: result.timestamp,
      message: 'Message sent successfully'
    });
  } catch (err) {
    console.error('Error sending message:', err);
    res.status(500).json({ error: 'Failed to send message', details: err.message });
  }
});

// GET /api/messages/:sessionId/list - Get conversation messages
router.get('/:sessionId/list', async (req, res) => {
  try {
    const { phone, limit = 50, offset = 0 } = req.query;

    const messages = await getSessionMessages(req.sessionId, parseInt(limit), parseInt(offset));
    
    // Filter by phone if provided
    let filtered = messages;
    if (phone) {
      filtered = messages.filter(m => m.from_phone === phone || m.to_phone === phone);
    }

    res.json({
      success: true,
      total: filtered.length,
      messages: filtered
    });
  } catch (err) {
    console.error('Error getting messages:', err);
    res.status(500).json({ error: 'Failed to get messages', details: err.message });
  }
});

// GET /api/messages/:sessionId/conversation - Get conversation with a specific contact
router.get('/:sessionId/conversation/:phone', async (req, res) => {
  try {
    const { limit = 50, offset = 0 } = req.query;
    const phone = req.params.phone;

    const messages = await getSessionMessages(req.sessionId, parseInt(limit), parseInt(offset));
    
    const conversation = messages.filter(m => 
      m.from_phone === phone || m.to_phone === phone || 
      m.from_phone.includes(phone) || m.to_phone.includes(phone)
    );

    res.json({
      success: true,
      phone: phone,
      total: conversation.length,
      messages: conversation
    });
  } catch (err) {
    console.error('Error getting conversation:', err);
    res.status(500).json({ error: 'Failed to get conversation', details: err.message });
  }
});

// GET /api/messages/:sessionId/contacts - Get all contacts
router.get('/:sessionId/contacts', async (req, res) => {
  try {
    const contacts = await getSessionContacts(req.sessionId);

    res.json({
      success: true,
      total: contacts.length,
      contacts: contacts
    });
  } catch (err) {
    console.error('Error getting contacts:', err);
    res.status(500).json({ error: 'Failed to get contacts', details: err.message });
  }
});

module.exports = router;
