const express = require('express');
const router = express.Router();
const { 
  registerWebhook, 
  getSessionWebhooks, 
  getWebhook, 
  updateWebhook, 
  deleteWebhook,
  testWebhook,
  getWebhookLogs
} = require('../webhooks');
const { validateApiKey } = require('../auth');

// Middleware to validate API key
async function validateApiKeyMiddleware(req, res, next) {
  const apiKey = req.headers['x-api-key'];
  const sessionId = req.params.sessionId;

  if (!apiKey || !sessionId) {
    return res.status(401).json({ error: 'Missing API key or session ID' });
  }

  const isValid = await validateApiKey(apiKey, sessionId);
  if (!isValid) {
    return res.status(401).json({ error: 'Invalid API key or session not authenticated' });
  }

  req.sessionId = sessionId;
  req.apiKey = apiKey;
  next();
}

// Apply middleware to all routes
router.use('/:sessionId', validateApiKeyMiddleware);

/**
 * POST /api/webhooks/:sessionId
 * Register a new webhook
 * 
 * Body:
 * {
 *   "eventType": "message.received|message.sent|contact.added|contact.updated|contact.deleted",
 *   "webhookUrl": "https://your-platform.com/webhook",
 *   "isActive": true
 * }
 */
router.post('/:sessionId', async (req, res) => {
  try {
    const { eventType, webhookUrl, isActive = true } = req.body;

    if (!eventType || !webhookUrl) {
      return res.status(400).json({ 
        error: 'Missing required fields',
        required: ['eventType', 'webhookUrl']
      });
    }

    const validEvents = [
      'message.received',
      'message.sent',
      'contact.added',
      'contact.updated',
      'contact.deleted',
      'session.authenticated',
      'session.disconnected'
    ];

    if (!validEvents.includes(eventType)) {
      return res.status(400).json({ 
        error: 'Invalid event type',
        validEvents: validEvents
      });
    }

    const webhook = await registerWebhook(req.sessionId, eventType, webhookUrl, isActive);

    res.status(201).json({
      success: true,
      webhook: webhook
    });
  } catch (err) {
    console.error('Error registering webhook:', err);
    res.status(500).json({ error: 'Failed to register webhook', details: err.message });
  }
});

/**
 * GET /api/webhooks/:sessionId
 * Get all webhooks for a session
 */
router.get('/:sessionId', async (req, res) => {
  try {
    const webhooks = await getSessionWebhooks(req.sessionId);

    res.json({
      success: true,
      total: webhooks.length,
      webhooks: webhooks
    });
  } catch (err) {
    console.error('Error getting webhooks:', err);
    res.status(500).json({ error: 'Failed to get webhooks', details: err.message });
  }
});

/**
 * GET /api/webhooks/:sessionId/:webhookId
 * Get specific webhook details
 */
router.get('/:sessionId/:webhookId', async (req, res) => {
  try {
    const webhook = await getWebhook(req.params.webhookId, req.sessionId);

    if (!webhook) {
      return res.status(404).json({ error: 'Webhook not found' });
    }

    res.json({
      success: true,
      webhook: webhook
    });
  } catch (err) {
    console.error('Error getting webhook:', err);
    res.status(500).json({ error: 'Failed to get webhook', details: err.message });
  }
});

/**
 * PUT /api/webhooks/:sessionId/:webhookId
 * Update a webhook
 */
router.put('/:sessionId/:webhookId', async (req, res) => {
  try {
    const { eventType, webhookUrl, isActive } = req.body;
    const { webhookId } = req.params;

    const webhook = await getWebhook(webhookId, req.sessionId);
    if (!webhook) {
      return res.status(404).json({ error: 'Webhook not found' });
    }

    const updated = await updateWebhook(
      webhookId,
      req.sessionId,
      eventType || webhook.event_type,
      webhookUrl || webhook.webhook_url,
      isActive !== undefined ? isActive : webhook.isActive
    );

    res.json({
      success: true,
      webhook: updated
    });
  } catch (err) {
    console.error('Error updating webhook:', err);
    res.status(500).json({ error: 'Failed to update webhook', details: err.message });
  }
});

/**
 * DELETE /api/webhooks/:sessionId/:webhookId
 * Delete a webhook
 */
router.delete('/:sessionId/:webhookId', async (req, res) => {
  try {
    const webhook = await getWebhook(req.params.webhookId, req.sessionId);
    if (!webhook) {
      return res.status(404).json({ error: 'Webhook not found' });
    }

    await deleteWebhook(req.params.webhookId, req.sessionId);

    res.json({
      success: true,
      message: 'Webhook deleted successfully'
    });
  } catch (err) {
    console.error('Error deleting webhook:', err);
    res.status(500).json({ error: 'Failed to delete webhook', details: err.message });
  }
});

/**
 * POST /api/webhooks/:sessionId/:webhookId/test
 * Test a webhook by sending a test payload
 */
router.post('/:sessionId/:webhookId/test', async (req, res) => {
  try {
    const webhook = await getWebhook(req.params.webhookId, req.sessionId);
    if (!webhook) {
      return res.status(404).json({ error: 'Webhook not found' });
    }

    const result = await testWebhook(req.params.webhookId, req.sessionId);

    res.json({
      success: result.status === 'success',
      result: result
    });
  } catch (err) {
    console.error('Error testing webhook:', err);
    res.status(500).json({ error: 'Failed to test webhook', details: err.message });
  }
});

/**
 * GET /api/webhooks/:sessionId/:webhookId/logs
 * Get webhook delivery logs
 */
router.get('/:sessionId/:webhookId/logs', async (req, res) => {
  try {
    const webhook = await getWebhook(req.params.webhookId, req.sessionId);
    if (!webhook) {
      return res.status(404).json({ error: 'Webhook not found' });
    }

    const { limit = 50 } = req.query;
    const logs = await getWebhookLogs(req.params.webhookId, parseInt(limit));

    res.json({
      success: true,
      total: logs.length,
      logs: logs
    });
  } catch (err) {
    console.error('Error getting webhook logs:', err);
    res.status(500).json({ error: 'Failed to get webhook logs', details: err.message });
  }
});

module.exports = router;
