const { v4: uuidv4 } = require('uuid');
const { runQuery, getQuery, allQuery } = require('./database');
const axios = require('axios');

/**
 * Webhook Management System
 * Handles webhook registration, triggering, and management
 */

// Register a webhook for a session
async function registerWebhook(sessionId, eventType, webhookUrl, isActive = true) {
  try {
    const webhookId = uuidv4();
    
    await runQuery(
      `INSERT INTO webhooks (id, session_id, event_type, webhook_url, is_active, created_at)
       VALUES (?, ?, ?, ?, ?, ?)`,
      [webhookId, sessionId, eventType, webhookUrl, isActive ? 1 : 0, new Date()]
    );

    return {
      webhookId,
      eventType,
      webhookUrl,
      isActive,
      status: 'registered'
    };
  } catch (err) {
    console.error('Error registering webhook:', err);
    throw err;
  }
}

// Get all webhooks for a session
async function getSessionWebhooks(sessionId) {
  try {
    const webhooks = await allQuery(
      'SELECT * FROM webhooks WHERE session_id = ? ORDER BY created_at DESC',
      [sessionId]
    );
    return webhooks.map(w => ({
      ...w,
      isActive: w.is_active === 1
    }));
  } catch (err) {
    console.error('Error getting webhooks:', err);
    throw err;
  }
}

// Get specific webhook
async function getWebhook(webhookId, sessionId) {
  try {
    const webhook = await getQuery(
      'SELECT * FROM webhooks WHERE id = ? AND session_id = ?',
      [webhookId, sessionId]
    );
    return webhook ? { ...webhook, isActive: webhook.is_active === 1 } : null;
  } catch (err) {
    console.error('Error getting webhook:', err);
    throw err;
  }
}

// Update webhook
async function updateWebhook(webhookId, sessionId, eventType, webhookUrl, isActive) {
  try {
    await runQuery(
      `UPDATE webhooks 
       SET event_type = ?, webhook_url = ?, is_active = ?, updated_at = ?
       WHERE id = ? AND session_id = ?`,
      [eventType, webhookUrl, isActive ? 1 : 0, new Date(), webhookId, sessionId]
    );

    return {
      webhookId,
      eventType,
      webhookUrl,
      isActive,
      status: 'updated'
    };
  } catch (err) {
    console.error('Error updating webhook:', err);
    throw err;
  }
}

// Delete webhook
async function deleteWebhook(webhookId, sessionId) {
  try {
    await runQuery(
      'DELETE FROM webhooks WHERE id = ? AND session_id = ?',
      [webhookId, sessionId]
    );

    return { status: 'deleted' };
  } catch (err) {
    console.error('Error deleting webhook:', err);
    throw err;
  }
}

// Test webhook by sending ping
async function testWebhook(webhookId, sessionId) {
  try {
    const webhook = await getWebhook(webhookId, sessionId);
    
    if (!webhook) {
      throw new Error('Webhook not found');
    }

    const testPayload = {
      event: 'webhook.test',
      timestamp: new Date(),
      sessionId: sessionId,
      message: 'This is a test webhook payload'
    };

    const response = await axios.post(webhook.webhook_url, testPayload, {
      timeout: 5000,
      headers: {
        'Content-Type': 'application/json',
        'User-Agent': 'WhatsApp-CRM-API/1.0'
      }
    });

    return {
      webhookId,
      status: 'success',
      statusCode: response.status,
      message: 'Webhook test successful'
    };
  } catch (err) {
    console.error('Error testing webhook:', err);
    return {
      webhookId,
      status: 'failed',
      error: err.message
    };
  }
}

// Trigger webhook event
async function triggerWebhook(sessionId, eventType, payload) {
  try {
    // Get all active webhooks for this session and event type
    const webhooks = await allQuery(
      `SELECT * FROM webhooks 
       WHERE session_id = ? AND event_type = ? AND is_active = 1`,
      [sessionId, eventType]
    );

    if (webhooks.length === 0) {
      return { triggered: 0, message: 'No active webhooks for this event' };
    }

    const webhookPayload = {
      event: eventType,
      timestamp: new Date().toISOString(),
      sessionId: sessionId,
      data: payload
    };

    let successCount = 0;
    let failureCount = 0;

    // Send webhook to all registered URLs
    for (const webhook of webhooks) {
      try {
        await axios.post(webhook.webhook_url, webhookPayload, {
          timeout: 10000,
          headers: {
            'Content-Type': 'application/json',
            'User-Agent': 'WhatsApp-CRM-API/1.0',
            'X-Webhook-ID': webhook.id
          }
        });

        // Log successful webhook
        await logWebhookDelivery(webhook.id, 200, 'Success');
        successCount++;
      } catch (err) {
        console.error(`Error triggering webhook ${webhook.id}:`, err.message);
        
        // Log failed webhook
        await logWebhookDelivery(webhook.id, err.response?.status || 0, err.message);
        failureCount++;
      }
    }

    return {
      triggered: webhooks.length,
      successCount,
      failureCount,
      event: eventType
    };
  } catch (err) {
    console.error('Error triggering webhooks:', err);
    throw err;
  }
}

// Log webhook delivery
async function logWebhookDelivery(webhookId, statusCode, message) {
  try {
    await runQuery(
      `INSERT INTO webhook_logs (webhook_id, status_code, message, timestamp)
       VALUES (?, ?, ?, ?)`,
      [webhookId, statusCode, message, new Date()]
    );
  } catch (err) {
    console.error('Error logging webhook delivery:', err);
  }
}

// Get webhook delivery logs
async function getWebhookLogs(webhookId, limit = 50) {
  try {
    const logs = await allQuery(
      `SELECT * FROM webhook_logs WHERE webhook_id = ? 
       ORDER BY timestamp DESC LIMIT ?`,
      [webhookId, limit]
    );
    return logs;
  } catch (err) {
    console.error('Error getting webhook logs:', err);
    throw err;
  }
}

module.exports = {
  registerWebhook,
  getSessionWebhooks,
  getWebhook,
  updateWebhook,
  deleteWebhook,
  testWebhook,
  triggerWebhook,
  logWebhookDelivery,
  getWebhookLogs
};
