const { Client, LocalAuth, MessageMedia } = require('whatsapp-web.js');
const QRCode = require('qrcode');
const { v4: uuidv4 } = require('uuid');
const path = require('path');
const fs = require('fs');
const { runQuery, getQuery, allQuery } = require('./database');
const { triggerWebhook } = require('./webhooks');

const clients = {}; // Store active clients
const qrCodes = {}; // Store QR codes temporarily

async function createWhatsAppSession(sessionId) {
  try {
    const sessionPath = path.join(process.cwd(), process.env.SESSION_PATH || './sessions');
    if (!fs.existsSync(sessionPath)) {
      fs.mkdirSync(sessionPath, { recursive: true });
    }

    const client = new Client({
      authStrategy: new LocalAuth({
        clientId: sessionId,
        dataPath: sessionPath
      }),
      puppeteer: {
        headless: true,
        args: ['--no-sandbox', '--disable-setuid-sandbox']
      }
    });

    // QR Code event
    client.on('qr', async (qr) => {
      console.log(`QR Code received for session: ${sessionId}`);
      qrCodes[sessionId] = qr;
      
      // Save QR code as data URL
      try {
        const qrDataUrl = await QRCode.toDataURL(qr);
        await runQuery(
          'UPDATE sessions SET qr_code = ? WHERE id = ?',
          [qrDataUrl, sessionId]
        );
      } catch (err) {
        console.error('Error saving QR code:', err);
      }
    });

    // Ready event
    client.on('ready', async () => {
      console.log(`Client ready for session: ${sessionId}`);
      const phoneInfo = client.info;
      
      await runQuery(
        'UPDATE sessions SET status = ?, phone_number = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?',
        ['authenticated', phoneInfo.me.user, sessionId]
      );
      
      clients[sessionId] = client;
      delete qrCodes[sessionId];

      // Trigger webhook for session authenticated
      try {
        await triggerWebhook(sessionId, 'session.authenticated', {
          phone_number: phoneInfo.me.user,
          timestamp: new Date()
        });
      } catch (err) {
        console.error('Error triggering webhook:', err);
      }
    });

    // Message event
    client.on('message', async (message) => {
      try {
        const contact = await message.getContact();
        const sender = message.from;
        const recipientId = message.to;
        
        // Save message to database
        await runQuery(
          `INSERT INTO messages 
          (id, session_id, from_phone, to_phone, message_text, message_type, direction, timestamp, media_url)
          VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)`,
          [
            message.id.id,
            sessionId,
            sender,
            recipientId,
            message.body || '',
            message.type,
            'incoming',
            new Date(message.timestamp * 1000),
            message.hasMedia ? message.downloadMedia().url : null
          ]
        );

        // Trigger webhook for message received
        try {
          await triggerWebhook(sessionId, 'message.received', {
            messageId: message.id.id,
            from: sender,
            to: recipientId,
            text: message.body,
            type: message.type,
            timestamp: new Date(message.timestamp * 1000)
          });
        } catch (err) {
          console.error('Error triggering webhook:', err);
        }

        // Save contact if not exists
        const existingContact = await getQuery(
          'SELECT id FROM contacts WHERE phone = ?',
          [sender]
        );

        if (!existingContact) {
          const contactId = uuidv4();
          await runQuery(
            `INSERT INTO contacts (id, session_id, phone, name)
            VALUES (?, ?, ?, ?)`,
            [contactId, sessionId, sender, contact.name || sender]
          );

          // Trigger webhook for contact added
          try {
            await triggerWebhook(sessionId, 'contact.added', {
              contactId: contactId,
              phone: sender,
              name: contact.name || sender
            });
          } catch (err) {
            console.error('Error triggering webhook:', err);
          }
        }

        console.log(`Message received from ${sender}: ${message.body}`);
      } catch (err) {
        console.error('Error processing message:', err);
      }
    });

    // Disconnected event
    client.on('disconnected', async (reason) => {
      console.log(`Client disconnected for session: ${sessionId}`, reason);
      await runQuery(
        'UPDATE sessions SET status = ? WHERE id = ?',
        ['disconnected', sessionId]
      );
      delete clients[sessionId];

      // Trigger webhook for session disconnected
      try {
        await triggerWebhook(sessionId, 'session.disconnected', {
          reason: reason,
          timestamp: new Date()
        });
      } catch (err) {
        console.error('Error triggering webhook:', err);
      }
    });

    // Initialize client
    await client.initialize();

    return {
      sessionId,
      status: 'pending',
      message: 'Client initialized, waiting for QR code authentication'
    };
  } catch (err) {
    console.error('Error creating WhatsApp session:', err);
    throw err;
  }
}

async function sendMessage(sessionId, recipientPhone, messageText, mediaPath = null) {
  try {
    const client = clients[sessionId];
    if (!client) {
      throw new Error('Session not authenticated');
    }

    // Format phone number
    const chatId = recipientPhone.includes('@') ? recipientPhone : `${recipientPhone}@c.us`;

    let messageResult;
    if (mediaPath && fs.existsSync(mediaPath)) {
      const media = MessageMedia.fromFilePath(mediaPath);
      messageResult = await client.sendMessage(chatId, media, { caption: messageText });
    } else {
      messageResult = await client.sendMessage(chatId, messageText);
    }

    // Save sent message to database
    await runQuery(
      `INSERT INTO messages 
      (id, session_id, from_phone, to_phone, message_text, message_type, direction, timestamp)
      VALUES (?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        messageResult.id.id,
        sessionId,
        client.info.me.user,
        chatId,
        messageText,
        'text',
        'outgoing',
        new Date()
      ]
    );

    // Trigger webhook for message sent
    try {
      await triggerWebhook(sessionId, 'message.sent', {
        messageId: messageResult.id.id,
        to: recipientPhone,
        text: messageText,
        timestamp: new Date()
      });
    } catch (err) {
      console.error('Error triggering webhook:', err);
    }

    return {
      success: true,
      messageId: messageResult.id.id,
      timestamp: new Date()
    };
  } catch (err) {
    console.error('Error sending message:', err);
    throw err;
  }
}

async function getSessionStatus(sessionId) {
  try {
    const session = await getQuery(
      'SELECT * FROM sessions WHERE id = ?',
      [sessionId]
    );

    if (!session) {
      return null;
    }

    return {
      id: session.id,
      phone_number: session.phone_number,
      status: session.status,
      qr_code: session.qr_code,
      api_key: session.api_key,
      created_at: session.created_at
    };
  } catch (err) {
    console.error('Error getting session status:', err);
    throw err;
  }
}

async function getSessionMessages(sessionId, limit = 50, offset = 0) {
  try {
    const messages = await allQuery(
      `SELECT * FROM messages WHERE session_id = ?
       ORDER BY timestamp DESC LIMIT ? OFFSET ?`,
      [sessionId, limit, offset]
    );
    return messages;
  } catch (err) {
    console.error('Error getting messages:', err);
    throw err;
  }
}

async function getSessionContacts(sessionId) {
  try {
    const contacts = await allQuery(
      'SELECT * FROM contacts WHERE session_id = ? ORDER BY name',
      [sessionId]
    );
    return contacts;
  } catch (err) {
    console.error('Error getting contacts:', err);
    throw err;
  }
}

async function logoutSession(sessionId) {
  try {
    const client = clients[sessionId];
    if (client) {
      await client.logout();
      delete clients[sessionId];
    }

    await runQuery(
      'UPDATE sessions SET status = ? WHERE id = ?',
      ['logged_out', sessionId]
    );

    return { success: true, message: 'Session logged out' };
  } catch (err) {
    console.error('Error logging out session:', err);
    throw err;
  }
}

module.exports = {
  createWhatsAppSession,
  sendMessage,
  getSessionStatus,
  getSessionMessages,
  getSessionContacts,
  logoutSession,
  clients,
  qrCodes
};
