# WhatsApp CRM API - Complete Guide

## Table of Contents

1. [Architecture Overview](#architecture-overview)
2. [Installation Guide](#installation-guide)
3. [Configuration](#configuration)
4. [API Reference](#api-reference)
5. [Authentication Flow](#authentication-flow)
6. [Usage Examples](#usage-examples)
7. [Advanced Features](#advanced-features)
8. [Deployment](#deployment)

---

## Architecture Overview

### System Components

```
┌─────────────────────────────────────────────────────────┐
│                     Client Application                   │
│              (Web/Mobile/Desktop CRM)                    │
└────────────────────┬────────────────────────────────────┘
                     │
                     │ HTTP/HTTPS
                     │
┌────────────────────▼────────────────────────────────────┐
│              WhatsApp CRM API Server                      │
│  ┌──────────────────────────────────────────────────┐  │
│  │         Express.js Application Server             │  │
│  │  ├─ Authentication Routes (/api/auth)            │  │
│  │  ├─ Message Routes (/api/messages)               │  │
│  │  ├─ Contact Routes (/api/contacts)               │  │
│  │  └─ Template Routes (/api/templates)             │  │
│  └──────────────────────────────────────────────────┘  │
│  ┌──────────────────────────────────────────────────┐  │
│  │   WhatsApp Web Client (whatsapp-web.js)          │  │
│  │   - QR Code Authentication                        │  │
│  │   - Real-time Message Listening                   │  │
│  │   - Message Sending                               │  │
│  │   - Contact Synchronization                       │  │
│  └──────────────────────────────────────────────────┘  │
└────────────────────┬────────────────────────────────────┘
                     │
                     │ WhatsApp Web
                     │
┌────────────────────▼────────────────────────────────────┐
│            WhatsApp Web Infrastructure                   │
│         (WhatsApp Message Service)                       │
└─────────────────────────────────────────────────────────┘

┌─────────────────────────────────────────────────────────┐
│              SQLite Database                             │
│  ┌─────────────┬─────────────┬──────────┬────────────┐ │
│  │  Sessions   │  Messages   │Contacts  │ Templates  │ │
│  └─────────────┴─────────────┴──────────┴────────────┘ │
└─────────────────────────────────────────────────────────┘
```

### Data Flow

1. **Authentication Flow**
   - User initiates session → API creates QR code → User scans → WhatsApp authenticates
   - API key is generated and returned → Used for subsequent requests

2. **Message Flow**
   - Incoming messages from WhatsApp → Listener captures → Stored in database
   - Outgoing messages via API → Sent through WhatsApp Web → Stored in database

3. **Contact Sync**
   - Contacts from WhatsApp → Synced with local database
   - New contacts can be added manually via API

---

## Installation Guide

### Prerequisites

- **Node.js**: v14.0.0 or higher
- **npm**: v6.0.0 or higher
- **Operating System**: Windows, macOS, or Linux
- **RAM**: Minimum 512MB, recommended 2GB

### Step-by-Step Installation

#### 1. Clone/Setup Project

```bash
# Navigate to the project directory
cd c:\Users\HP\Documents\whatsappcrm

# Or create a new directory
mkdir whatsappcrm
cd whatsappcrm
```

#### 2. Initialize Node Project

```bash
# If no package.json exists
npm init -y
```

#### 3. Install Dependencies

```bash
npm install
```

This will install:
- **express** - Web framework
- **whatsapp-web.js** - WhatsApp Web client
- **qrcode** - QR code generation
- **sqlite3** - Database
- **dotenv** - Environment variables
- **cors** - Cross-origin requests
- **uuid** - ID generation
- **body-parser** - Request parsing
- **axios** - HTTP client (optional, for external APIs)
- **moment** - Date/time utilities

#### 4. Verify Installation

```bash
node -v
npm -v
npm list
```

---

## Configuration

### Environment Variables (.env)

```env
# Server Configuration
PORT=3000
NODE_ENV=development

# API Configuration
API_KEY=your_secret_api_key_change_this
JWT_SECRET=your_jwt_secret_key

# Database Configuration
DATABASE_PATH=./data/whatsapp.db

# Session Configuration
SESSION_PATH=./sessions
LOG_PATH=./logs

# WhatsApp Configuration
WHATSAPP_BROWSER=Chrome
WHATSAPP_HEADLESS=true

# CORS Configuration
CORS_ORIGIN=*

# Rate Limiting
RATE_LIMIT_WINDOW=15
RATE_LIMIT_MAX_REQUESTS=100
```

### Database Schema

The SQLite database is automatically created with the following tables:

**Sessions Table**
```sql
CREATE TABLE sessions (
  id TEXT PRIMARY KEY,
  phone_number TEXT UNIQUE,
  api_key TEXT UNIQUE,
  qr_code TEXT,
  status TEXT DEFAULT 'pending',
  created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
);
```

**Messages Table**
```sql
CREATE TABLE messages (
  id TEXT PRIMARY KEY,
  session_id TEXT,
  from_phone TEXT,
  to_phone TEXT,
  message_text TEXT,
  message_type TEXT DEFAULT 'text',
  direction TEXT DEFAULT 'incoming',
  timestamp DATETIME DEFAULT CURRENT_TIMESTAMP,
  read_status INTEGER DEFAULT 0,
  media_url TEXT,
  FOREIGN KEY(session_id) REFERENCES sessions(id)
);
```

**Contacts Table**
```sql
CREATE TABLE contacts (
  id TEXT PRIMARY KEY,
  session_id TEXT,
  phone TEXT UNIQUE NOT NULL,
  name TEXT,
  avatar TEXT,
  created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY(session_id) REFERENCES sessions(id)
);
```

**Templates Table**
```sql
CREATE TABLE templates (
  id TEXT PRIMARY KEY,
  session_id TEXT,
  name TEXT,
  content TEXT,
  variables TEXT,
  created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
  FOREIGN KEY(session_id) REFERENCES sessions(id)
);
```

---

## API Reference

### Base URL
```
http://localhost:3000/api
```

### Authentication Header
```
X-API-Key: your_api_key_here
```

### Response Format
All responses are in JSON format:

**Success Response:**
```json
{
  "success": true,
  "data": { /* response data */ }
}
```

**Error Response:**
```json
{
  "error": "Error Type",
  "message": "Detailed error message"
}
```

### Status Codes
- `200` - OK
- `201` - Created
- `202` - Accepted
- `400` - Bad Request
- `401` - Unauthorized
- `404` - Not Found
- `500` - Internal Server Error

---

## Authentication Flow

### 1. Initialize Session

**Endpoint:** `POST /api/auth/init`

**No authentication required**

**Response:**
```json
{
  "success": true,
  "sessionId": "uuid-string",
  "apiKey": "long-api-key-string",
  "status": "pending",
  "message": "Session initialized. Wait for QR code and scan it with your WhatsApp.",
  "qrCodeUrl": "/api/auth/uuid-string/qr"
}
```

### 2. Get QR Code

**Endpoint:** `GET /api/auth/:sessionId/qr`

**No authentication required**

**Response (QR Pending):**
```json
{
  "status": "pending",
  "message": "QR code not yet generated. Please wait..."
}
```

**Response (QR Ready):**
```json
{
  "status": "pending",
  "qr_code": "data:image/png;base64,..."
}
```

### 3. Check Authentication Status

**Endpoint:** `GET /api/auth/:sessionId/status`

**Header:** `X-API-Key: your_api_key`

**Response:**
```json
{
  "success": true,
  "sessionId": "uuid-string",
  "status": "authenticated",
  "phone_number": "919876543210",
  "api_key": "long-api-key-string",
  "created_at": "2024-01-30T10:00:00Z"
}
```

---

## Usage Examples

### JavaScript/Node.js

#### Using the Provided Client

```javascript
const WhatsAppCRMClient = require('./client');

async function example() {
  const client = new WhatsAppCRMClient('http://localhost:3000');

  // Initialize
  const session = await client.initSession();
  console.log('Session:', session);

  // Get QR code
  const qr = await client.getQRCode();
  console.log('QR Code:', qr.qr_code);

  // Wait for authentication
  const status = await client.waitForAuthentication();
  console.log('Authenticated:', status);

  // Send message
  const result = await client.sendMessage('919876543210', 'Hello!');
  console.log('Message sent:', result);
}

example();
```

### cURL Examples

#### Initialize Session
```bash
curl -X POST http://localhost:3000/api/auth/init
```

#### Get QR Code
```bash
curl http://localhost:3000/api/auth/YOUR_SESSION_ID/qr
```

#### Check Status
```bash
curl http://localhost:3000/api/auth/YOUR_SESSION_ID/status \
  -H "X-API-Key: YOUR_API_KEY"
```

#### Send Message
```bash
curl -X POST http://localhost:3000/api/messages/YOUR_SESSION_ID/send \
  -H "X-API-Key: YOUR_API_KEY" \
  -H "Content-Type: application/json" \
  -d '{
    "phone": "919876543210",
    "message": "Hello World!"
  }'
```

#### Add Contact
```bash
curl -X POST http://localhost:3000/api/contacts/YOUR_SESSION_ID \
  -H "X-API-Key: YOUR_API_KEY" \
  -H "Content-Type: application/json" \
  -d '{
    "phone": "919876543210",
    "name": "John Doe"
  }'
```

### Python Example

```python
import requests
import json

BASE_URL = "http://localhost:3000/api"

# Initialize session
response = requests.post(f"{BASE_URL}/auth/init")
session = response.json()
session_id = session['sessionId']
api_key = session['apiKey']

print(f"Session ID: {session_id}")
print(f"API Key: {api_key}")

# Send message
headers = {"X-API-Key": api_key}
payload = {
    "phone": "919876543210",
    "message": "Hello from Python!"
}

response = requests.post(
    f"{BASE_URL}/messages/{session_id}/send",
    headers=headers,
    json=payload
)

print(response.json())
```

---

## Advanced Features

### Message Templates

Templates allow you to create reusable message patterns with variables.

**Create Template:**
```javascript
const template = await client.createTemplate(
  'Order Confirmation',
  'Thank you {{name}}! Your order {{orderId}} has been confirmed.',
  ['name', 'orderId']
);
```

**Render Template:**
```javascript
const rendered = await client.renderTemplate(template.templateId, {
  name: 'John Doe',
  orderId: 'ORD-12345'
});

console.log(rendered.rendered); 
// "Thank you John Doe! Your order ORD-12345 has been confirmed."
```

**Send with Template:**
```javascript
await client.sendMessage(
  '919876543210',
  rendered.rendered
);
```

### Batch Operations

**Send Messages to Multiple Contacts:**
```javascript
const contacts = ['919876543210', '919876543211', '919876543212'];

for (const phone of contacts) {
  await client.sendMessage(phone, 'Bulk message!');
  // Add delay to avoid rate limiting
  await new Promise(r => setTimeout(r, 1000));
}
```

### Message Filtering

**Get Messages by Contact:**
```javascript
const conversation = await client.getConversation('919876543210');
console.log(conversation.messages);
```

**Get Recent Messages:**
```javascript
const messages = await client.getMessages(50, 0);
```

---

## Deployment

### Local Development

```bash
npm install
npm run dev
```

Server runs at: `http://localhost:3000`

### Production Deployment

#### Using PM2

```bash
# Install PM2 globally
npm install -g pm2

# Start application
pm2 start src/index.js --name "whatsapp-crm"

# Monitor
pm2 monit

# View logs
pm2 logs whatsapp-crm

# Restart
pm2 restart whatsapp-crm

# Stop
pm2 stop whatsapp-crm
```

#### Using Docker

**Dockerfile:**
```dockerfile
FROM node:16-alpine

WORKDIR /app

COPY package*.json ./
RUN npm ci --only=production

COPY . .

EXPOSE 3000

CMD ["npm", "start"]
```

**Build and Run:**
```bash
docker build -t whatsapp-crm-api .
docker run -p 3000:3000 whatsapp-crm-api
```

#### Using Nginx Reverse Proxy

```nginx
server {
    listen 80;
    server_name your-domain.com;

    location / {
        proxy_pass http://localhost:3000;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_cache_bypass $http_upgrade;
    }
}
```

### Environment Setup for Production

Update `.env`:
```env
PORT=3000
NODE_ENV=production
API_KEY=generate-strong-random-key
DATABASE_PATH=/var/lib/whatsapp-crm/database.db
SESSION_PATH=/var/lib/whatsapp-crm/sessions
LOG_PATH=/var/log/whatsapp-crm
```

---

## Troubleshooting

| Issue | Solution |
|-------|----------|
| QR Code not loading | Wait 10-15 seconds, check Node.js logs |
| Connection refused | Check PORT in .env, verify server is running |
| Invalid API Key | Verify header name is exactly `X-API-Key` |
| Database locked | Ensure only one instance is running |
| WhatsApp logout | Re-scan QR code to re-authenticate |
| Memory leak | Update whatsapp-web.js to latest version |

---

## Security Best Practices

1. **Change default API key** in `.env`
2. **Use HTTPS** in production
3. **Implement rate limiting** on production servers
4. **Secure database** with proper permissions
5. **Validate** all user inputs
6. **Use firewall** to restrict API access
7. **Rotate API keys** regularly
8. **Monitor logs** for suspicious activity
9. **Use environment variables** for sensitive data
10. **Implement authentication** for admin functions

---

## Performance Optimization

1. **Database Indexes:** Add indexes on frequently queried columns
2. **Caching:** Implement Redis for session caching
3. **Load Balancing:** Use multiple server instances
4. **Message Queue:** Use RabbitMQ/Redis for async operations
5. **Connection Pooling:** Optimize database connections

---

## License

ISC
