# WhatsApp CRM API - Production Ready

A robust, enterprise-grade WhatsApp CRM API with QR code authentication, admin dashboard, API key management, and webhook support.

## ✨ Features

✅ **Admin Authentication** - Secure login with token-based session management  
✅ **QR Code Authentication** - Easy WhatsApp Web connection via QR scanning  
✅ **API Key System** - Unique keys for each session, easy API access  
✅ **Message Management** - Send and receive WhatsApp messages  
✅ **Contact Management** - Store and manage WhatsApp contacts  
✅ **Message Templates** - Reusable templates with variable support  
✅ **Webhook Support** - Real-time event notifications  
✅ **SQLite Database** - Persistent storage with cascading deletes  
✅ **Error Handling** - Comprehensive error codes and messages  
✅ **Production Ready** - Proper middleware, logging, and validation  

## 🚀 Quick Start

### Prerequisites
- Node.js 14+
- npm or yarn

### Installation

```bash
# 1. Navigate to project
cd whatsappcrm

# 2. Install dependencies
npm install

# 3. Create .env file
cp .env.example .env

# 4. Start server
npm start
```

Server runs on `http://localhost:3000`

## 📊 Architecture

### Two-Tier Authentication System

**Admin Login** → **Admin Token** → **Session Management** → **API Key** → **API Operations**

## 📱 Admin Dashboard

Access at: `http://localhost:3000`

**Default Credentials:**
- Username: `admin`
- Password: `admin123`

## 🔐 Quick API Examples

### Get Admin Token
```bash
curl -X POST http://localhost:3000/api/auth/admin/login \
  -H "Content-Type: application/json" \
  -d '{"username":"admin","password":"admin123"}'
```

### Send Message
```bash
curl -X POST http://localhost:3000/api/messages/{sessionId}/send \
  -H "X-API-Key: your_api_key" \
  -H "Content-Type: application/json" \
  -d '{"phone":"5511999999999","message":"Hello!"}'
```

## 📡 Key Endpoints

**Admin:**
- `POST /api/auth/admin/login` - Login
- `POST /api/auth/init` - Initialize session
- `GET /api/auth/:sessionId/qr` - Get QR code

**Messages:**
- `POST /api/messages/:sessionId/send` - Send message
- `GET /api/messages/:sessionId/list` - Get messages

**Contacts:**
- `POST /api/contacts/:sessionId` - Add contact
- `GET /api/contacts/:sessionId` - List contacts

**Webhooks:**
- `POST /api/webhooks/:sessionId` - Register webhook
- `GET /api/webhooks/:sessionId` - List webhooks

## 💾 Database

SQLite database auto-created in `./data/whatsapp.db`

**Tables:**
- sessions
- contacts
- messages
- templates
- webhooks
- webhook_logs

## 📂 Project Structure

```
whatsappcrm/
├── src/
│   ├── middleware/      # Authentication & error handling
│   ├── routes/          # API endpoints
│   ├── auth.js          # Session & key management
│   ├── database.js      # SQLite setup
│   ├── whatsapp.js      # WhatsApp client
│   ├── webhooks.js      # Webhook management
│   └── index.js         # Server entry
├── public/
│   ├── login.html       # Admin login page
│   ├── dashboard.html   # Admin dashboard
│   └── styles.css       # Styling
├── data/                # Database (auto-created)
├── sessions/            # WhatsApp sessions (auto-created)
└── logs/                # Logs (auto-created)
```

## ⚙️ Configuration

Edit `.env`:
```env
PORT=3000
NODE_ENV=development
ADMIN_USERNAME=admin
ADMIN_PASSWORD=admin123
DATABASE_PATH=./data/whatsapp.db
SESSION_PATH=./sessions
LOG_PATH=./logs
```

## 🛡️ Security Features

- Token expiration (24 hours)
- API key validation on all endpoints
- Session isolation
- Input validation
- CORS protection
- SQL injection prevention
- Account lockout (5 attempts)

## 🐛 Error Codes

- `MISSING_CREDENTIALS` - Missing auth
- `INVALID_CREDENTIALS` - Wrong credentials
- `INVALID_API_KEY` - Invalid API key
- `NOT_FOUND` - Resource not found
- `INVALID_ADMIN_TOKEN` - Token expired

## 📚 Full Documentation

- [SETUP_GUIDE.md](SETUP_GUIDE.md) - Detailed setup
- `/api/docs` - API documentation
- `/health` - Health check

## 🚨 Troubleshooting

| Issue | Solution |
|-------|----------|
| Port in use | Change PORT in .env |
| DB locked | Restart server |
| Invalid API Key | Verify key matches session |
| Token expired | Login again |

## 📊 Health Check

```bash
curl http://localhost:3000/health
```

## 📝 Development

```bash
npm install       # Install dependencies
npm run dev       # Start with auto-reload
npm start         # Production mode
```

## 📦 Dependencies

- express, whatsapp-web.js, sqlite3, qrcode, uuid, cors, body-parser

## 📄 License

ISC

## 👨‍💻 Support

- Full API docs: http://localhost:3000/api/docs
- Setup guide: [SETUP_GUIDE.md](SETUP_GUIDE.md)
- Check error codes for specific issues

---

**Version**: 1.0.0  
**Status**: Production Ready ✅

Or for development with auto-reload:
```bash
npm run dev
```

The server will start at `http://localhost:3000`

## Quick Start

### 1. Initialize a Session

```bash
curl -X POST http://localhost:3000/api/auth/init
```

Response:
```json
{
  "success": true,
  "sessionId": "uuid-here",
  "apiKey": "long-api-key-here",
  "status": "pending",
  "message": "Session initialized. Wait for QR code and scan it with your WhatsApp.",
  "qrCodeUrl": "/api/auth/uuid-here/qr"
}
```

### 2. Get QR Code

```bash
curl http://localhost:3000/api/auth/YOUR_SESSION_ID/qr
```

This returns a QR code as data URL. Open it in your browser and scan with WhatsApp.

### 3. Check Authentication Status

```bash
curl http://localhost:3000/api/auth/YOUR_SESSION_ID/status \
  -H "X-API-Key: YOUR_API_KEY"
```

### 4. Send a Message

```bash
curl -X POST http://localhost:3000/api/messages/YOUR_SESSION_ID/send \
  -H "X-API-Key: YOUR_API_KEY" \
  -H "Content-Type: application/json" \
  -d '{
    "phone": "919876543210",
    "message": "Hello from WhatsApp CRM API!"
  }'
```

### 5. Get Contacts

```bash
curl http://localhost:3000/api/contacts/YOUR_SESSION_ID \
  -H "X-API-Key: YOUR_API_KEY"
```

## API Endpoints

### Authentication

| Method | Endpoint | Description |
|--------|----------|-------------|
| POST | `/api/auth/init` | Initialize a new session with QR code |
| GET | `/api/auth/:sessionId/qr` | Get QR code for scanning |
| GET | `/api/auth/:sessionId/status` | Check session status |

### Messages

| Method | Endpoint | Description |
|--------|----------|-------------|
| POST | `/api/messages/:sessionId/send` | Send a message |
| GET | `/api/messages/:sessionId/list` | Get all messages |
| GET | `/api/messages/:sessionId/conversation/:phone` | Get conversation with a contact |

### Contacts

| Method | Endpoint | Description |
|--------|----------|-------------|
| POST | `/api/contacts/:sessionId` | Add a new contact |
| GET | `/api/contacts/:sessionId` | Get all contacts |
| GET | `/api/contacts/:sessionId/:phone` | Get specific contact |
| PUT | `/api/contacts/:sessionId/:id` | Update a contact |
| DELETE | `/api/contacts/:sessionId/:id` | Delete a contact |

### Templates

| Method | Endpoint | Description |
|--------|----------|-------------|
| POST | `/api/templates/:sessionId` | Create a template |
| GET | `/api/templates/:sessionId` | Get all templates |
| GET | `/api/templates/:sessionId/:id` | Get specific template |
| PUT | `/api/templates/:sessionId/:id` | Update a template |
| POST | `/api/templates/:sessionId/:id/render` | Render template with variables |
| DELETE | `/api/templates/:sessionId/:id` | Delete a template |

## Authentication

All endpoints (except `/auth/init` and `/auth/:sessionId/qr`) require authentication using API Key:

```
Header: X-API-Key: your_api_key_here
```

## Example Use Cases

### Create a Template

```bash
curl -X POST http://localhost:3000/api/templates/YOUR_SESSION_ID \
  -H "X-API-Key: YOUR_API_KEY" \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Welcome Message",
    "content": "Hello {{name}}, welcome to our service!",
    "variables": ["name"]
  }'
```

### Render Template

```bash
curl -X POST http://localhost:3000/api/templates/YOUR_SESSION_ID/TEMPLATE_ID/render \
  -H "X-API-Key: YOUR_API_KEY" \
  -H "Content-Type: application/json" \
  -d '{
    "variables": {
      "name": "John Doe"
    }
  }'
```

### Send Using Template

```bash
# First render the template
# Then send the rendered message
curl -X POST http://localhost:3000/api/messages/YOUR_SESSION_ID/send \
  -H "X-API-Key: YOUR_API_KEY" \
  -H "Content-Type: application/json" \
  -d '{
    "phone": "919876543210",
    "message": "Hello John Doe, welcome to our service!"
  }'
```

## Database Schema

### Sessions Table
- `id` - Unique session identifier
- `phone_number` - WhatsApp phone number (after authentication)
- `api_key` - Unique API key for this session
- `qr_code` - QR code data URL
- `status` - Session status (pending, authenticated, disconnected)
- `created_at` - Creation timestamp
- `updated_at` - Last update timestamp

### Messages Table
- `id` - Message ID
- `session_id` - Associated session
- `from_phone` - Sender's phone number
- `to_phone` - Recipient's phone number
- `message_text` - Message content
- `message_type` - Type (text, media, etc.)
- `direction` - incoming or outgoing
- `timestamp` - Message timestamp
- `read_status` - Whether message was read
- `media_url` - URL if message contains media

### Contacts Table
- `id` - Contact ID
- `session_id` - Associated session
- `phone` - Contact's phone number
- `name` - Contact's name
- `avatar` - Contact's avatar URL
- `created_at` - When contact was added

### Templates Table
- `id` - Template ID
- `session_id` - Associated session
- `name` - Template name
- `content` - Template content with {{variables}}
- `variables` - JSON array of variable names
- `created_at` - Creation timestamp

## Error Handling

All errors are returned with appropriate HTTP status codes:

```json
{
  "error": "Error Type",
  "details": "Detailed error message"
}
```

## Security Notes

1. ⚠️ Change the default API_KEY in `.env`
2. ⚠️ Use HTTPS in production
3. ⚠️ Keep API keys secure
4. ⚠️ Don't expose session IDs publicly
5. ⚠️ Implement rate limiting in production

## Project Structure

```
whatsappcrm/
├── src/
│   ├── index.js              # Main server file
│   ├── database.js           # Database operations
│   ├── auth.js               # Authentication logic
│   ├── whatsapp.js           # WhatsApp Web integration
│   └── routes/
│       ├── auth.js           # Auth endpoints
│       ├── messages.js       # Message endpoints
│       ├── contacts.js       # Contact endpoints
│       └── templates.js      # Template endpoints
├── data/                     # SQLite database
├── sessions/                 # WhatsApp sessions
├── logs/                     # Application logs
├── .env                      # Environment variables
├── package.json              # Dependencies
└── README.md                 # This file
```

## Development

For development with hot reload:

```bash
npm install --save-dev nodemon
npm run dev
```

## Production Deployment

1. Set `NODE_ENV=production` in `.env`
2. Use a process manager like PM2:
```bash
npm install -g pm2
pm2 start src/index.js --name "whatsapp-crm-api"
```
3. Set up reverse proxy (nginx/Apache)
4. Use HTTPS certificates
5. Implement rate limiting and authentication

## Troubleshooting

### QR Code Not Showing
- Wait 10-15 seconds after initialization
- Check that Node.js process is running
- Check console for any errors

### Messages Not Receiving
- Ensure WhatsApp is not logged in elsewhere
- Check that session is authenticated
- Verify phone number format (include country code)

### API Key Errors
- Verify API key is passed in `X-API-Key` header
- Check that session is authenticated first
- Ensure session ID matches the API key

## License

ISC

## Support

For issues and questions, please create an issue in the repository.
