# WhatsApp CRM - Setup Guide

## Overview

This is a stable, production-ready WhatsApp CRM API with:
- ✅ Admin login system
- ✅ QR code session authentication  
- ✅ API key management
- ✅ Message & Contact management
- ✅ Webhook support
- ✅ SQLite database
- ✅ Comprehensive error handling

## Quick Start

### 1. Installation

```bash
# Navigate to project directory
cd whatsappcrm

# Install dependencies
npm install
```

### 2. Environment Setup

Create a `.env` file in the project root:

```env
PORT=3000
NODE_ENV=development

# Admin Credentials (Change these in production!)
ADMIN_USERNAME=admin
ADMIN_PASSWORD=admin123

# Database
DATABASE_PATH=./data/whatsapp.db
SESSION_PATH=./sessions
LOG_PATH=./logs
```

### 3. Start the Server

```bash
# Development mode (with auto-reload)
npm run dev

# Production mode
npm start
```

Server will start on `http://localhost:3000`

## Access Points

### Admin Dashboard
- **URL**: http://localhost:3000
- **Login**: admin / admin123
- **Features**: 
  - Create new WhatsApp sessions
  - View QR codes for scanning
  - Manage sessions
  - Get API keys

### API Documentation
- **URL**: http://localhost:3000/api/docs
- **Contains**: Full endpoint reference with authentication details

### Health Check
- **URL**: http://localhost:3000/health
- **Returns**: Server status and uptime

## Authentication System

### Two-Tier Authentication

#### 1. Admin Token (for session management)
Required for:
- Creating new WhatsApp sessions
- Viewing QR codes
- Managing sessions

**Login Endpoint:**
```bash
POST /api/auth/admin/login
Content-Type: application/json

{
  "username": "admin",
  "password": "admin123"
}
```

**Response:**
```json
{
  "success": true,
  "token": "eyJhbGciOiJIUzI1NiIs...",
  "expiresAt": "2024-01-31T10:00:00Z",
  "user": { "username": "admin" }
}
```

#### 2. API Key (for API operations)
Required for:
- Sending messages
- Fetching contacts
- Managing templates
- Webhook operations

**How to Get:**
1. Login to admin dashboard
2. Create a new session
3. Copy the API Key from "Session Status" card

## File Structure

```
whatsappcrm/
├── src/
│   ├── middleware/
│   │   ├── auth.js          # Authentication middleware
│   │   └── error.js         # Error handling
│   ├── routes/
│   │   ├── auth.js          # Auth endpoints
│   │   ├── messages.js      # Message endpoints
│   │   ├── contacts.js      # Contact endpoints
│   │   ├── templates.js     # Template endpoints
│   │   └── webhooks.js      # Webhook endpoints
│   ├── auth.js              # Session & API key logic
│   ├── database.js          # SQLite management
│   ├── whatsapp.js          # WhatsApp client
│   ├── webhooks.js          # Webhook management
│   └── index.js             # Server entry point
├── public/
│   ├── login.html           # Admin login page
│   ├── dashboard.html       # Admin dashboard
│   ├── styles.css           # Styles
│   └── dashboard.js         # Dashboard logic
├── data/                    # Database files (created automatically)
├── sessions/                # WhatsApp sessions (created automatically)
├── logs/                    # Log files (created automatically)
├── .env                     # Environment variables
├── .env.example             # Example env file
└── package.json             # Dependencies
```

## API Examples

### Send a Message

```bash
curl -X POST http://localhost:3000/api/messages/{sessionId}/send \
  -H "X-API-Key: your_api_key" \
  -H "Content-Type: application/json" \
  -d '{
    "phone": "5511999999999",
    "message": "Hello from WhatsApp CRM!"
  }'
```

### Get Contacts

```bash
curl http://localhost:3000/api/contacts/{sessionId} \
  -H "X-API-Key: your_api_key"
```

### Register Webhook

```bash
curl -X POST http://localhost:3000/api/webhooks/{sessionId} \
  -H "X-API-Key: your_api_key" \
  -H "Content-Type: application/json" \
  -d '{
    "event_type": "message.received",
    "webhook_url": "https://your-domain.com/webhook"
  }'
```

## Webhook Events

Supported events:
- `message.received` - New message received
- `message.sent` - Message sent successfully
- `contact.added` - New contact added
- `session.authenticated` - Session authenticated
- `session.disconnected` - Session disconnected

## Key Features

### 1. Session Management
- Each WhatsApp session gets a unique Session ID
- Each session has a unique API Key for authentication
- Sessions are stored in SQLite database

### 2. QR Code Authentication
- Generated on-demand when initializing a session
- Displayed in admin dashboard
- Scanned with WhatsApp to authenticate

### 3. Security
- Admin token-based access to sensitive endpoints
- API key validation for all API operations
- Rate limiting ready (can be added via middleware)
- Environment variable protection

### 4. Error Handling
- Comprehensive error messages with error codes
- Proper HTTP status codes
- Stack traces in development mode

## Troubleshooting

### Issue: "Session not found"
- Make sure the Session ID is correct
- Check if the session was properly initialized
- Refresh the dashboard

### Issue: "Invalid API Key"
- Verify you're using the correct API Key
- Make sure the API Key matches the session ID
- Copy the key directly from the dashboard

### Issue: "Admin token expired"
- Login again to get a new token
- Tokens are valid for 24 hours
- Clear browser cache and try again

### Issue: Database errors
- Ensure write permissions in project directory
- Check if `./data` directory exists
- Try deleting the database and reinitializing

## Production Deployment

Before deploying to production:

1. **Change admin credentials**
   ```env
   ADMIN_USERNAME=your_secure_username
   ADMIN_PASSWORD=your_secure_password
   ```

2. **Use proper secret management**
   - Store credentials in secrets manager
   - Use environment variables for sensitive data

3. **Enable HTTPS**
   - Use a reverse proxy (Nginx, Apache)
   - Install SSL certificates

4. **Database backup**
   - Regular backups of SQLite database
   - Consider migrating to PostgreSQL for production

5. **Logging**
   - Enable structured logging
   - Monitor error rates

6. **Rate limiting**
   - Add rate limiting middleware
   - Protect against abuse

## Support & Updates

- Check `/api/docs` for full API documentation
- Review error messages for specific issues
- Check application logs in `./logs` directory

## License

ISC

---

**Version**: 1.0.0  
**Last Updated**: January 30, 2024
