/**
 * Example usage of WhatsApp CRM API Client
 */

const WhatsAppCRMClient = require('./client');

async function main() {
  const client = new WhatsAppCRMClient('http://localhost:3000');

  try {
    console.log('=== WhatsApp CRM API Example ===\n');

    // Step 1: Initialize session
    console.log('1. Initializing session...');
    const initResponse = await client.initSession();
    console.log('Session ID:', initResponse.sessionId);
    console.log('API Key:', initResponse.apiKey);
    console.log('QR Code URL:', initResponse.qrCodeUrl, '\n');

    // Step 2: Get QR code
    console.log('2. Getting QR code...');
    const qrResponse = await client.getQRCode();
    if (qrResponse.qr_code) {
      console.log('QR Code generated! Status:', qrResponse.status);
      // In a real application, display the QR code image
    } else {
      console.log('Waiting for QR code generation...', qrResponse.message);
    }

    // Step 3: Wait for authentication
    console.log('\n3. Waiting for authentication (max 60 seconds)...');
    console.log('Please scan the QR code with your WhatsApp mobile app...\n');
    
    try {
      const authStatus = await client.waitForAuthentication();
      console.log('✓ Authentication successful!');
      console.log('Phone Number:', authStatus.phone_number);
      console.log('Status:', authStatus.status, '\n');

      // Step 4: Add some contacts
      console.log('4. Adding contacts...');
      const contact1 = await client.addContact('919876543210', 'John Doe');
      console.log('Contact added:', contact1);

      // Step 5: Send a message
      console.log('\n5. Sending a message...');
      const messageResult = await client.sendMessage(
        '919876543210',
        'Hello from WhatsApp CRM API! This is a test message.'
      );
      console.log('Message sent:', messageResult);

      // Step 6: Create a template
      console.log('\n6. Creating a message template...');
      const template = await client.createTemplate(
        'Greeting Template',
        'Hello {{name}}, welcome to our service! Your order ID is {{orderId}}.',
        ['name', 'orderId']
      );
      console.log('Template created:', template);

      // Step 7: Render template
      console.log('\n7. Rendering template...');
      const rendered = await client.renderTemplate(template.templateId, {
        name: 'John Doe',
        orderId: 'ORD-12345'
      });
      console.log('Rendered message:', rendered.rendered);

      // Step 8: Get all contacts
      console.log('\n8. Getting all contacts...');
      const allContacts = await client.getContacts();
      console.log('Total contacts:', allContacts.total);
      console.log('Contacts:', allContacts.contacts);

      // Step 9: Get messages
      console.log('\n9. Getting messages...');
      const messages = await client.getMessages(10);
      console.log('Total messages:', messages.total);
      console.log('Latest messages:', messages.messages.slice(0, 3));

      console.log('\n=== Example completed successfully! ===');
    } catch (authError) {
      console.log('Authentication timeout. Please make sure to scan the QR code.');
      console.log('You can try again with the same session ID and API key.');
    }
  } catch (error) {
    console.error('Error:', error.message);
  }
}

// Run example
if (require.main === module) {
  main();
}

module.exports = main;
