// Configuration
const API_BASE_URL = 'http://localhost:3000/api';
const SESSION_STORAGE_KEY = 'whatsapp_crm_session';
const WEBHOOK_STORAGE_KEY = 'whatsapp_crm_webhook';

// State Management
let currentSession = null;
let currentUser = null;

// Initialize on page load
document.addEventListener('DOMContentLoaded', () => {
    checkAuthentication();
    setupEventListeners();
});

// Event Listeners Setup
function setupEventListeners() {
    // Login form
    const loginForm = document.getElementById('login-form');
    loginForm.addEventListener('submit', handleLogin);

    // Logout button
    const logoutBtn = document.getElementById('logout-btn');
    logoutBtn.addEventListener('click', handleLogout);

    // Copy buttons
    document.getElementById('copy-session-btn').addEventListener('click', () => copyToClipboard('session-id'));
    document.getElementById('copy-webhook-btn').addEventListener('click', () => copyToClipboard('webhook-id'));

    // Refresh status
    document.getElementById('refresh-status-btn').addEventListener('click', checkConnectionStatus);

    // Webhook creation
    document.getElementById('create-webhook-btn').addEventListener('click', createWebhook);

    // Navigation links
    document.querySelectorAll('.nav-link').forEach(link => {
        link.addEventListener('click', (e) => {
            e.preventDefault();
            const target = link.getAttribute('href');
            document.querySelector(target).scrollIntoView({ behavior: 'smooth' });
        });
    });
}

// Authentication Functions
function checkAuthentication() {
    const session = localStorage.getItem(SESSION_STORAGE_KEY);
    
    if (session) {
        const sessionData = JSON.parse(session);
        currentSession = sessionData.sessionId;
        currentUser = sessionData.username;
        showDashboard();
        initializeQRCode();
        checkConnectionStatus();
        loadWebhooks();
    } else {
        showLogin();
    }
}

async function handleLogin(e) {
    e.preventDefault();

    const username = document.getElementById('username').value;
    const password = document.getElementById('password').value;
    const errorDiv = document.getElementById('login-error');
    const successDiv = document.getElementById('login-success');

    // Clear previous messages
    errorDiv.classList.remove('show');
    successDiv.classList.remove('show');

    // Validate credentials (Default: admin/admin)
    if (username === 'admin' && password === 'admin') {
        // Generate session ID and Webhook ID
        const sessionId = generateId();
        const webhookId = generateId();

        // Initialize WhatsApp session
        try {
            const response = await fetch(`${API_BASE_URL}/auth/init`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ sessionId })
            });

            if (response.ok) {
                // Save session to localStorage
                const sessionData = {
                    sessionId,
                    webhookId,
                    username,
                    loginTime: new Date().toISOString()
                };

                localStorage.setItem(SESSION_STORAGE_KEY, JSON.stringify(sessionData));
                localStorage.setItem(WEBHOOK_STORAGE_KEY, JSON.stringify({ webhookId }));

                currentSession = sessionId;
                currentUser = username;

                // Show success message
                successDiv.textContent = 'Login successful!';
                successDiv.classList.add('show');

                // Show dashboard after short delay
                setTimeout(() => {
                    showDashboard();
                    initializeQRCode();
                    checkConnectionStatus();
                }, 500);
            } else {
                throw new Error('Failed to initialize session');
            }
        } catch (error) {
            console.error('Login error:', error);
            errorDiv.textContent = 'Failed to initialize session. Please try again.';
            errorDiv.classList.add('show');
        }
    } else {
        // Invalid credentials
        errorDiv.textContent = 'Invalid username or password';
        errorDiv.classList.add('show');
    }
}

function handleLogout() {
    if (confirm('Are you sure you want to logout?')) {
        localStorage.removeItem(SESSION_STORAGE_KEY);
        localStorage.removeItem(WEBHOOK_STORAGE_KEY);
        currentSession = null;
        currentUser = null;
        showLogin();
        document.getElementById('login-form').reset();
    }
}

// UI Functions
function showLogin() {
    document.getElementById('login-container').classList.add('visible');
    document.querySelector('.dashboard-container').classList.remove('visible');
}

function showDashboard() {
    document.getElementById('login-container').classList.remove('visible');
    document.querySelector('.dashboard-container').classList.add('visible');
}

// QR Code Functions
async function initializeQRCode() {
    try {
        const qrDiv = document.getElementById('qr-code');
        const sessionIdSpan = document.getElementById('session-id');
        const webhookIdSpan = document.getElementById('webhook-id');

        // Display session and webhook IDs
        sessionIdSpan.textContent = currentSession;
        
        const webhookData = JSON.parse(localStorage.getItem(WEBHOOK_STORAGE_KEY));
        webhookIdSpan.textContent = webhookData?.webhookId || 'Not generated';

        // Fetch QR code
        const response = await fetch(`${API_BASE_URL}/auth/${currentSession}/qr`, {
            method: 'GET'
        });

        if (response.ok) {
            const data = await response.json();
            
            if (data.qr) {
                qrDiv.innerHTML = `<img src="${data.qr}" alt="QR Code">`;
            } else {
                // QR not ready yet, retry after delay
                qrDiv.innerHTML = '<p class="loading">Generating QR Code</p>';
                setTimeout(initializeQRCode, 2000);
            }
        } else {
            qrDiv.innerHTML = '<p style="color: red;">Failed to load QR code</p>';
        }
    } catch (error) {
        console.error('Error loading QR code:', error);
        document.getElementById('qr-code').innerHTML = '<p style="color: red;">Error loading QR code</p>';
    }
}

// Connection Status Functions
async function checkConnectionStatus() {
    try {
        const response = await fetch(`${API_BASE_URL}/auth/${currentSession}/status`, {
            method: 'GET'
        });

        if (response.ok) {
            const data = await response.json();
            updateConnectionUI(data);
        }
    } catch (error) {
        console.error('Error checking connection status:', error);
    }
}

function updateConnectionUI(data) {
    const statusElement = document.getElementById('connection-status');
    const statusConnectedCard = document.getElementById('status-connected');
    const statusDetail = document.getElementById('status-detail');
    const accountDetail = document.getElementById('account-detail');
    const sessionDetail = document.getElementById('session-detail');

    if (data.authenticated) {
        statusElement.textContent = 'Connected';
        statusElement.className = 'status connected';
        statusConnectedCard.classList.remove('status-disconnected');
        statusConnectedCard.classList.add('status-connected');
        statusDetail.textContent = 'WhatsApp is connected';

        if (data.phoneNumber) {
            accountDetail.textContent = `Phone: ${data.phoneNumber}`;
        }

        sessionDetail.textContent = `Started: ${new Date(data.timestamp).toLocaleDateString()}`;
    } else {
        statusElement.textContent = 'Disconnected';
        statusElement.className = 'status disconnected';
        statusConnectedCard.classList.remove('status-connected');
        statusConnectedCard.classList.add('status-disconnected');
        statusDetail.textContent = 'Scan QR code to connect';
        accountDetail.textContent = 'No account connected';
        sessionDetail.textContent = '-';

        // Retry QR initialization if not authenticated
        setTimeout(initializeQRCode, 3000);
    }
}

// Webhook Functions
async function createWebhook() {
    const url = document.getElementById('webhook-url').value.trim();
    const messageDiv = document.getElementById('webhook-message');

    if (!url) {
        showMessage(messageDiv, 'Please enter a webhook URL', 'error');
        return;
    }

    try {
        // Get selected events
        const selectedEvents = Array.from(document.querySelectorAll('.checkbox-group input:checked'))
            .map(checkbox => checkbox.value);

        if (selectedEvents.length === 0) {
            showMessage(messageDiv, 'Please select at least one event', 'error');
            return;
        }

        const response = await fetch(`${API_BASE_URL}/webhooks/${currentSession}`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                url,
                events: selectedEvents
            })
        });

        if (response.ok) {
            const data = await response.json();
            showMessage(messageDiv, 'Webhook created successfully!', 'success');
            
            // Clear form
            document.getElementById('webhook-url').value = '';
            document.querySelectorAll('.checkbox-group input').forEach(cb => cb.checked = false);
            
            // Reload webhooks
            setTimeout(loadWebhooks, 1000);
        } else {
            showMessage(messageDiv, 'Failed to create webhook', 'error');
        }
    } catch (error) {
        console.error('Error creating webhook:', error);
        showMessage(messageDiv, 'Error creating webhook', 'error');
    }
}

async function loadWebhooks() {
    try {
        const response = await fetch(`${API_BASE_URL}/webhooks/${currentSession}`, {
            method: 'GET'
        });

        if (response.ok) {
            const data = await response.json();
            const webhooksContainer = document.getElementById('webhooks-container');

            if (data.webhooks && data.webhooks.length > 0) {
                webhooksContainer.innerHTML = data.webhooks.map(webhook => `
                    <div class="webhook-item">
                        <div class="webhook-item-info">
                            <h4>${webhook.url}</h4>
                            <p>Events: ${webhook.events.join(', ')}</p>
                            <p>Created: ${new Date(webhook.createdAt).toLocaleDateString()}</p>
                        </div>
                        <div class="webhook-item-actions">
                            <button class="btn btn-small" onclick="testWebhook('${webhook.id}')">Test</button>
                            <button class="btn btn-small btn-danger" onclick="deleteWebhook('${webhook.id}')">Delete</button>
                        </div>
                    </div>
                `).join('');
            } else {
                webhooksContainer.innerHTML = '<p class="empty-state">No webhooks configured</p>';
            }
        }
    } catch (error) {
        console.error('Error loading webhooks:', error);
    }
}

async function deleteWebhook(webhookId) {
    if (!confirm('Are you sure you want to delete this webhook?')) return;

    try {
        const response = await fetch(`${API_BASE_URL}/webhooks/${currentSession}/${webhookId}`, {
            method: 'DELETE'
        });

        if (response.ok) {
            loadWebhooks();
        }
    } catch (error) {
        console.error('Error deleting webhook:', error);
    }
}

async function testWebhook(webhookId) {
    try {
        const response = await fetch(`${API_BASE_URL}/webhooks/${currentSession}/${webhookId}/test`, {
            method: 'POST'
        });

        if (response.ok) {
            alert('Test webhook sent successfully!');
        }
    } catch (error) {
        console.error('Error testing webhook:', error);
        alert('Failed to send test webhook');
    }
}

// Utility Functions
function generateId() {
    return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function(c) {
        const r = Math.random() * 16 | 0;
        const v = c === 'x' ? r : (r & 0x3 | 0x8);
        return v.toString(16);
    });
}

function copyToClipboard(elementId) {
    const element = document.getElementById(elementId);
    const text = element.textContent;

    navigator.clipboard.writeText(text).then(() => {
        const button = event.target;
        const originalText = button.textContent;
        button.textContent = 'Copied!';
        setTimeout(() => {
            button.textContent = originalText;
        }, 2000);
    }).catch(error => {
        console.error('Copy failed:', error);
    });
}

function showMessage(element, message, type) {
    element.textContent = message;
    element.className = `message show ${type}`;
    
    setTimeout(() => {
        element.classList.remove('show');
    }, 5000);
}

// Auto-refresh connection status every 10 seconds when not authenticated
setInterval(() => {
    if (currentSession && document.querySelector('.dashboard-container').classList.contains('visible')) {
        checkConnectionStatus();
    }
}, 10000);
