const sqlite3 = require('sqlite3').verbose();
const path = require('path');
const fs = require('fs');

let db;

function initializeDatabase(dbPath) {
  return new Promise((resolve, reject) => {
    // Ensure directory exists
    const dir = path.dirname(dbPath);
    if (!fs.existsSync(dir)) {
      fs.mkdirSync(dir, { recursive: true });
    }

    db = new sqlite3.Database(dbPath, (err) => {
      if (err) {
        console.error('Failed to open database:', err);
        reject(err);
      } else {
        console.log(`Database opened: ${dbPath}`);
        createTables().then(resolve).catch(reject);
      }
    });

    // Enable foreign keys
    db.run('PRAGMA foreign_keys = ON');
  });
}

function createTables() {
  return new Promise((resolve, reject) => {
    db.serialize(() => {
      let tableCount = 0;
      const totalTables = 7;
      let errors = [];

      const onTableCreated = () => {
        tableCount++;
        if (tableCount === totalTables) {
          if (errors.length > 0) {
            reject(new Error(`Failed to create tables: ${errors.join(', ')}`));
          } else {
            console.log('All tables created successfully');
            resolve();
          }
        }
      };

      // Sessions table
      db.run(`
        CREATE TABLE IF NOT EXISTS sessions (
          id TEXT PRIMARY KEY,
          phone_number TEXT UNIQUE,
          api_key TEXT UNIQUE NOT NULL,
          qr_code TEXT,
          status TEXT DEFAULT 'pending' CHECK(status IN ('pending', 'authenticated', 'disconnected')),
          created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
          updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
        )
      `, (err) => {
        if (err) errors.push(`sessions: ${err.message}`);
        onTableCreated();
      });

      // Contacts table
      db.run(`
        CREATE TABLE IF NOT EXISTS contacts (
          id TEXT PRIMARY KEY,
          session_id TEXT NOT NULL,
          phone TEXT NOT NULL,
          name TEXT,
          avatar TEXT,
          created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
          UNIQUE(session_id, phone),
          FOREIGN KEY(session_id) REFERENCES sessions(id) ON DELETE CASCADE
        )
      `, (err) => {
        if (err) errors.push(`contacts: ${err.message}`);
        onTableCreated();
      });

      // Messages table
      db.run(`
        CREATE TABLE IF NOT EXISTS messages (
          id TEXT PRIMARY KEY,
          session_id TEXT NOT NULL,
          from_phone TEXT NOT NULL,
          to_phone TEXT NOT NULL,
          message_text TEXT,
          message_type TEXT DEFAULT 'text',
          direction TEXT DEFAULT 'incoming' CHECK(direction IN ('incoming', 'outgoing')),
          timestamp DATETIME DEFAULT CURRENT_TIMESTAMP,
          read_status INTEGER DEFAULT 0,
          media_url TEXT,
          FOREIGN KEY(session_id) REFERENCES sessions(id) ON DELETE CASCADE
        )
      `, (err) => {
        if (err) errors.push(`messages: ${err.message}`);
        onTableCreated();
      });

      // Templates table
      db.run(`
        CREATE TABLE IF NOT EXISTS templates (
          id TEXT PRIMARY KEY,
          session_id TEXT NOT NULL,
          name TEXT NOT NULL,
          content TEXT NOT NULL,
          variables TEXT,
          created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
          UNIQUE(session_id, name),
          FOREIGN KEY(session_id) REFERENCES sessions(id) ON DELETE CASCADE
        )
      `, (err) => {
        if (err) errors.push(`templates: ${err.message}`);
        onTableCreated();
      });

      // Webhooks table
      db.run(`
        CREATE TABLE IF NOT EXISTS webhooks (
          id TEXT PRIMARY KEY,
          session_id TEXT NOT NULL,
          event_type TEXT NOT NULL,
          webhook_url TEXT NOT NULL,
          is_active INTEGER DEFAULT 1,
          created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
          updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
          FOREIGN KEY(session_id) REFERENCES sessions(id) ON DELETE CASCADE
        )
      `, (err) => {
        if (err) errors.push(`webhooks: ${err.message}`);
        onTableCreated();
      });

      // Webhook logs table
      db.run(`
        CREATE TABLE IF NOT EXISTS webhook_logs (
          id INTEGER PRIMARY KEY AUTOINCREMENT,
          webhook_id TEXT NOT NULL,
          status_code INTEGER,
          response_time INTEGER,
          message TEXT,
          timestamp DATETIME DEFAULT CURRENT_TIMESTAMP,
          FOREIGN KEY(webhook_id) REFERENCES webhooks(id) ON DELETE CASCADE
        )
      `, (err) => {
        if (err) errors.push(`webhook_logs: ${err.message}`);
        onTableCreated();
      });

      // Admin tokens table (temporary, for demo)
      db.run(`
        CREATE TABLE IF NOT EXISTS admin_tokens (
          token TEXT PRIMARY KEY,
          username TEXT NOT NULL,
          expires_at INTEGER NOT NULL,
          created_at DATETIME DEFAULT CURRENT_TIMESTAMP
        )
      `, (err) => {
        if (err) errors.push(`admin_tokens: ${err.message}`);
        onTableCreated();
      });
    });
  });
}

function runQuery(query, params = []) {
  return new Promise((resolve, reject) => {
    db.run(query, params, (err) => {
      if (err) reject(err);
      else resolve();
    });
  });
}

function getQuery(query, params = []) {
  return new Promise((resolve, reject) => {
    db.get(query, params, (err, row) => {
      if (err) reject(err);
      else resolve(row);
    });
  });
}

function allQuery(query, params = []) {
  return new Promise((resolve, reject) => {
    db.all(query, params, (err, rows) => {
      if (err) reject(err);
      else resolve(rows || []);
    });
  });
}

module.exports = {
  initializeDatabase,
  runQuery,
  getQuery,
  allQuery
};
