require('dotenv').config();
const express = require('express');
const bodyParser = require('body-parser');
const cors = require('cors');
const path = require('path');
const fs = require('fs');

// Import modules
const { initializeDatabase } = require('./database');
const { requestLogger, errorHandler, notFoundHandler } = require('./middleware/error');

// Import routes
const authRoutes = require('./routes/auth');
const messagesRoutes = require('./routes/messages');
const contactsRoutes = require('./routes/contacts');
const templatesRoutes = require('./routes/templates');
const webhooksRoutes = require('./routes/webhooks');

const app = express();
const PORT = process.env.PORT || 3000;
const NODE_ENV = process.env.NODE_ENV || 'development';

const dirs = [
  process.env.SESSION_PATH || './sessions',
  process.env.LOG_PATH || './logs'
];

// Ensure DB directory (NOT DB file)
const dbPath = process.env.DATABASE_PATH || './data/whatsapp.db';
const dbDir = path.dirname(dbPath);

[...dirs, dbDir].forEach(dir => {
  if (!fs.existsSync(dir)) {
    fs.mkdirSync(dir, { recursive: true });
  }
});


// Middleware - Order matters!
app.use(requestLogger); // Log all requests
app.use(cors());
app.use(bodyParser.json({ limit: '50mb' }));
app.use(bodyParser.urlencoded({ limit: '50mb', extended: true }));

// Serve static files from public directory
const publicPath = path.join(__dirname, '../public');
if (fs.existsSync(publicPath)) {
  app.use(express.static(publicPath));
}

// Routes
app.use('/api/auth', authRoutes);
app.use('/api/messages', messagesRoutes);
app.use('/api/contacts', contactsRoutes);
app.use('/api/templates', templatesRoutes);
app.use('/api/webhooks', webhooksRoutes);

// Health check endpoint
app.get('/health', (req, res) => {
  res.json({
    status: 'ok',
    uptime: process.uptime(),
    timestamp: new Date().toISOString(),
    environment: NODE_ENV,
    message: 'WhatsApp CRM API is running'
  });
});

// API Documentation endpoint
app.get('/api/docs', (req, res) => {
  res.json({
    name: 'WhatsApp CRM API',
    version: '1.0.0',
    description: 'Complete WhatsApp CRM API with QR Authentication, Sessions, and Webhooks',
    baseUrl: `http://localhost:${PORT}/api`,
    authentication: {
      admin: {
        type: 'Token',
        description: 'Required for admin operations (QR, session management)',
        header: 'X-Admin-Token',
        endpoint: 'POST /api/auth/admin/login'
      },
      apiKey: {
        type: 'API Key',
        description: 'Required for API operations (messages, contacts, etc)',
        header: 'X-API-Key',
        note: 'Obtained when initializing a session'
      }
    },
    endpoints: {
      admin: {
        'POST /auth/admin/login': { 
          description: 'Admin login - get token',
          body: { username: 'string', password: 'string' }
        },
        'POST /auth/admin/logout': {
          description: 'Admin logout',
          auth: 'adminToken'
        },
        'GET /auth/admin/verify': {
          description: 'Verify admin token',
          auth: 'adminToken'
        },
        'POST /auth/init': {
          description: 'Initialize new WhatsApp session',
          auth: 'adminToken'
        },
        'GET /auth/:sessionId/qr': {
          description: 'Get QR code for session',
          auth: 'adminToken'
        },
        'GET /auth/sessions/list': {
          description: 'List all sessions',
          auth: 'adminToken'
        }
      },
      messages: {
        'POST /messages/:sessionId/send': {
          description: 'Send message',
          auth: 'apiKey',
          body: { phone: 'string', message: 'string', mediaPath: 'string?' }
        },
        'GET /messages/:sessionId/list': {
          description: 'Get messages',
          auth: 'apiKey'
        },
        'GET /messages/:sessionId/conversation/:phone': {
          description: 'Get conversation',
          auth: 'apiKey'
        }
      },
      contacts: {
        'POST /contacts/:sessionId': {
          description: 'Add contact',
          auth: 'apiKey'
        },
        'GET /contacts/:sessionId': {
          description: 'List contacts',
          auth: 'apiKey'
        },
        'PUT /contacts/:sessionId/:id': {
          description: 'Update contact',
          auth: 'apiKey'
        },
        'DELETE /contacts/:sessionId/:id': {
          description: 'Delete contact',
          auth: 'apiKey'
        }
      },
      webhooks: {
        'POST /webhooks/:sessionId': {
          description: 'Register webhook',
          auth: 'apiKey'
        },
        'GET /webhooks/:sessionId': {
          description: 'List webhooks',
          auth: 'apiKey'
        },
        'DELETE /webhooks/:sessionId/:webhookId': {
          description: 'Delete webhook',
          auth: 'apiKey'
        }
      }
    },
    webhookEvents: [
      'message.received',
      'message.sent',
      'contact.added',
      'session.authenticated',
      'session.disconnected'
    ]
  });
});

// Root route - serve admin dashboard
app.get('/', (req, res) => {
  const dashboardPath = path.join(publicPath, 'login.html');
  if (fs.existsSync(dashboardPath)) {
    res.sendFile(dashboardPath);
  } else {
    res.json({
      message: 'WhatsApp CRM API',
      endpoints: {
        dashboard: 'GET /',
        health: 'GET /health',
        docs: 'GET /api/docs',
        adminLogin: 'POST /api/auth/admin/login'
      }
    });
  }
});

// Error handlers - must be last!
app.use(notFoundHandler);
app.use(errorHandler);

// Initialize database and start server
async function startServer() {
  try {
    const dbPath = process.env.DATABASE_PATH || './data/whatsapp.db';
    await initializeDatabase(dbPath);
    console.log('Database initialized successfully');

    app.listen(PORT, () => {
      console.log(`
╔═══════════════════════════════════════════════════════╗
║        WhatsApp CRM API Server Started                ║
╠═══════════════════════════════════════════════════════╣
║  Server: http://localhost:${PORT}
║  Health Check: http://localhost:${PORT}/health
║  Documentation: http://localhost:${PORT}/api/docs
╚═══════════════════════════════════════════════════════╝
      `);
    });
  } catch (err) {
    console.error('Failed to start server:', err);
    process.exit(1);
  }
}

startServer();
