const crypto = require('crypto');

// Admin credentials (you should hash these in production)
const ADMIN_USERNAME = process.env.ADMIN_USERNAME || 'admin';
const ADMIN_PASSWORD = process.env.ADMIN_PASSWORD || 'admin123';

/**
 * Validate API Key Middleware
 * Checks if the provided API key is valid for the session
 */
async function validateApiKeyMiddleware(req, res, next) {
  const apiKey = req.headers['x-api-key'] || req.query.api_key;
  const sessionId = req.params.sessionId || req.body.sessionId;

  if (!apiKey || !sessionId) {
    return res.status(401).json({
      success: false,
      error: 'Missing API key or session ID',
      code: 'MISSING_CREDENTIALS'
    });
  }

  try {
    const { validateApiKey } = require('../auth');
    const isValid = await validateApiKey(apiKey, sessionId);

    if (!isValid) {
      return res.status(401).json({
        success: false,
        error: 'Invalid API key or session not authenticated',
        code: 'INVALID_CREDENTIALS'
      });
    }

    req.sessionId = sessionId;
    req.apiKey = apiKey;
    next();
  } catch (err) {
    console.error('API Key validation error:', err);
    res.status(500).json({
      success: false,
      error: 'Authentication validation failed',
      code: 'AUTH_ERROR'
    });
  }
}

/**
 * Validate Admin Credentials
 */
function validateAdminCredentials(username, password) {
  return username === ADMIN_USERNAME && password === ADMIN_PASSWORD;
}

/**
 * Generate secure token for admin session
 */
function generateAdminToken() {
  return crypto.randomBytes(32).toString('hex');
}

/**
 * Validate Admin Token Middleware
 */
function validateAdminToken(req, res, next) {
  const token = req.headers['x-admin-token'] || req.cookies?.adminToken;

  if (!token) {
    return res.status(401).json({
      success: false,
      error: 'Missing admin token',
      code: 'NO_ADMIN_TOKEN'
    });
  }

  // In production, you should store tokens in a cache/database with expiration
  const adminTokens = global.adminTokens || new Map();
  const tokenData = adminTokens.get(token);

  if (!tokenData || Date.now() > tokenData.expiresAt) {
    return res.status(401).json({
      success: false,
      error: 'Admin token expired or invalid',
      code: 'INVALID_ADMIN_TOKEN'
    });
  }

  req.adminToken = token;
  next();
}

module.exports = {
  validateApiKeyMiddleware,
  validateAdminCredentials,
  generateAdminToken,
  validateAdminToken,
  ADMIN_USERNAME
};
