const express = require('express');
const router = express.Router();
const { 
  createSession, 
  validateApiKey, 
  getSessionByApiKey 
} = require('../auth');
const { 
  createWhatsAppSession, 
  getSessionStatus 
} = require('../whatsapp');
const {
  validateAdminCredentials,
  generateAdminToken,
  validateAdminToken
} = require('../middleware/auth');
const { getQuery, allQuery } = require('../database');

/**
 * POST /api/auth/admin/login
 * Admin login endpoint - returns admin token for QR dashboard access
 */
router.post('/admin/login', async (req, res, next) => {
  try {
    const { username, password } = req.body;

    if (!username || !password) {
      return res.status(400).json({
        success: false,
        error: 'Username and password are required',
        code: 'MISSING_CREDENTIALS'
      });
    }

    if (!validateAdminCredentials(username, password)) {
      return res.status(401).json({
        success: false,
        error: 'Invalid username or password',
        code: 'INVALID_CREDENTIALS'
      });
    }

    // Generate admin token (valid for 24 hours)
    const token = generateAdminToken();
    const expiresAt = Date.now() + (24 * 60 * 60 * 1000);

    // Store token in memory (use Redis in production)
    if (!global.adminTokens) {
      global.adminTokens = new Map();
    }
    global.adminTokens.set(token, { expiresAt, username });

    res.json({
      success: true,
      message: 'Admin login successful',
      token,
      expiresAt: new Date(expiresAt).toISOString(),
      user: { username }
    });
  } catch (err) {
    next(err);
  }
});

/**
 * POST /api/auth/admin/logout
 * Admin logout endpoint
 */
router.post('/admin/logout', validateAdminToken, (req, res) => {
  if (!global.adminTokens) {
    return res.status(400).json({ success: false, error: 'No active sessions' });
  }

  global.adminTokens.delete(req.adminToken);

  res.json({
    success: true,
    message: 'Logged out successfully'
  });
});

/**
 * GET /api/auth/admin/verify
 * Verify admin token validity
 */
router.get('/admin/verify', (req, res, next) => {
  const token = req.headers['x-admin-token'];

  if (!token) {
    return res.status(401).json({
      success: false,
      error: 'No token provided'
    });
  }

  if (!global.adminTokens) {
    return res.status(401).json({
      success: false,
      error: 'Token invalid'
    });
  }

  const tokenData = global.adminTokens.get(token);

  if (!tokenData || Date.now() > tokenData.expiresAt) {
    return res.status(401).json({
      success: false,
      error: 'Token expired or invalid'
    });
  }

  res.json({
    success: true,
    valid: true,
    user: tokenData.username,
    expiresAt: new Date(tokenData.expiresAt).toISOString()
  });
});

/**
 * POST /api/auth/init
 * Initialize a new WhatsApp session with QR code
 * Requires admin token
 */
router.post('/init', validateAdminToken, async (req, res, next) => {
  try {
    const session = await createSession();
    const whatsappSession = await createWhatsAppSession(session.sessionId);

    res.json({
      success: true,
      sessionId: session.sessionId,
      apiKey: session.apiKey,
      status: 'pending',
      message: 'Session initialized. Scan the QR code with your WhatsApp.',
      qrCodeUrl: `/api/auth/${session.sessionId}/qr`
    });
  } catch (err) {
    next(err);
  }
});

/**
 * GET /api/auth/:sessionId/qr
 * Get QR code for a session
 * Returns QR code as data URL or image
 * Requires admin token
 */
router.get('/:sessionId/qr', validateAdminToken, async (req, res, next) => {
  try {
    const format = req.query.format || 'data'; // 'data' or 'image'
    const session = await getQuery(
      'SELECT * FROM sessions WHERE id = ?',
      [req.params.sessionId]
    );

    if (!session) {
      return res.status(404).json({
        success: false,
        error: 'Session not found',
        code: 'NOT_FOUND'
      });
    }

    if (format === 'image') {
      // Return QR code as image
      res.setHeader('Content-Type', 'image/png');
      res.send(session.qr_code);
    } else {
      // Return as data URL
      res.json({
        success: true,
        sessionId: req.params.sessionId,
        qrCode: session.qr_code,
        status: session.status
      });
    }
  } catch (err) {
    next(err);
  }
});

/**
 * GET /api/auth/:sessionId/status
 * Check session authentication status
 * Requires either API key or admin token
 */
router.get('/:sessionId/status', async (req, res, next) => {
  try {
    const apiKey = req.headers['x-api-key'];
    const adminToken = req.headers['x-admin-token'];

    // Check if either API key or admin token is provided
    if (apiKey) {
      const isValid = await validateApiKey(apiKey, req.params.sessionId);
      if (!isValid) {
        return res.status(401).json({
          success: false,
          error: 'Invalid API key',
          code: 'INVALID_API_KEY'
        });
      }
    } else if (adminToken) {
      if (!global.adminTokens) {
        return res.status(401).json({
          success: false,
          error: 'Invalid admin token'
        });
      }

      const tokenData = global.adminTokens.get(adminToken);
      if (!tokenData || Date.now() > tokenData.expiresAt) {
        return res.status(401).json({
          success: false,
          error: 'Admin token expired or invalid'
        });
      }
    } else {
      return res.status(401).json({
        success: false,
        error: 'Missing authentication credentials',
        code: 'NO_CREDENTIALS'
      });
    }

    const session = await getQuery(
      'SELECT * FROM sessions WHERE id = ?',
      [req.params.sessionId]
    );

    if (!session) {
      return res.status(404).json({
        success: false,
        error: 'Session not found',
        code: 'NOT_FOUND'
      });
    }

    res.json({
      success: true,
      sessionId: session.id,
      status: session.status,
      phoneNumber: session.phone_number || null,
      createdAt: session.created_at,
      updatedAt: session.updated_at
    });
  } catch (err) {
    next(err);
  }
});

/**
 * GET /api/auth/sessions/list
 * List all active sessions
 * Requires admin token
 */
router.get('/sessions/list', validateAdminToken, async (req, res, next) => {
  try {
    const sessions = await allQuery(
      'SELECT id, phone_number, status, created_at, updated_at FROM sessions ORDER BY created_at DESC'
    );

    res.json({
      success: true,
      total: sessions.length,
      sessions
    });
  } catch (err) {
    next(err);
  }
});

module.exports = router;
