const express = require('express');
const router = express.Router();
const { v4: uuidv4 } = require('uuid');
const { runQuery, getQuery, allQuery } = require('../database');
const { validateApiKey } = require('../auth');
const { triggerWebhook } = require('../webhooks');

// Middleware to validate API key
async function validateApiKeyMiddleware(req, res, next) {
  const apiKey = req.headers['x-api-key'];
  const sessionId = req.params.sessionId;

  if (!apiKey || !sessionId) {
    return res.status(401).json({ error: 'Missing API key or session ID' });
  }

  const isValid = await validateApiKey(apiKey, sessionId);
  if (!isValid) {
    return res.status(401).json({ error: 'Invalid API key or session not authenticated' });
  }

  req.sessionId = sessionId;
  req.apiKey = apiKey;
  next();
}

// Apply middleware to all routes
router.use('/:sessionId', validateApiKeyMiddleware);

// POST /api/contacts/:sessionId - Create/add a contact
router.post('/:sessionId', async (req, res) => {
  try {
    const { phone, name } = req.body;

    if (!phone) {
      return res.status(400).json({ error: 'Phone is required' });
    }

    const contactId = uuidv4();
    await runQuery(
      `INSERT INTO contacts (id, session_id, phone, name)
       VALUES (?, ?, ?, ?)`,
      [contactId, req.sessionId, phone, name || phone]
    );

    // Trigger webhook for contact added
    try {
      await triggerWebhook(req.sessionId, 'contact.added', {
        contactId: contactId,
        phone: phone,
        name: name || phone,
        timestamp: new Date()
      });
    } catch (err) {
      console.error('Error triggering webhook:', err);
    }

    res.json({
      success: true,
      contactId: contactId,
      phone: phone,
      name: name || phone
    });
  } catch (err) {
    console.error('Error creating contact:', err);
    res.status(500).json({ error: 'Failed to create contact', details: err.message });
  }
});

// GET /api/contacts/:sessionId - Get all contacts
router.get('/:sessionId', async (req, res) => {
  try {
    const contacts = await allQuery(
      'SELECT * FROM contacts WHERE session_id = ? ORDER BY name',
      [req.sessionId]
    );

    res.json({
      success: true,
      total: contacts.length,
      contacts: contacts
    });
  } catch (err) {
    console.error('Error getting contacts:', err);
    res.status(500).json({ error: 'Failed to get contacts', details: err.message });
  }
});

// GET /api/contacts/:sessionId/:phone - Get a specific contact
router.get('/:sessionId/:phone', async (req, res) => {
  try {
    const contact = await getQuery(
      'SELECT * FROM contacts WHERE session_id = ? AND phone LIKE ?',
      [req.sessionId, `%${req.params.phone}%`]
    );

    if (!contact) {
      return res.status(404).json({ error: 'Contact not found' });
    }

    res.json({
      success: true,
      contact: contact
    });
  } catch (err) {
    console.error('Error getting contact:', err);
    res.status(500).json({ error: 'Failed to get contact', details: err.message });
  }
});

// PUT /api/contacts/:sessionId/:id - Update a contact
router.put('/:sessionId/:id', async (req, res) => {
  try {
    const { name, phone } = req.body;
    const { id } = req.params;

    await runQuery(
      `UPDATE contacts SET name = ?, phone = ? WHERE id = ? AND session_id = ?`,
      [name, phone, id, req.sessionId]
    );

    // Trigger webhook for contact updated
    try {
      await triggerWebhook(req.sessionId, 'contact.updated', {
        contactId: id,
        phone: phone,
        name: name,
        timestamp: new Date()
      });
    } catch (err) {
      console.error('Error triggering webhook:', err);
    }

    res.json({
      success: true,
      message: 'Contact updated successfully'
    });
  } catch (err) {
    console.error('Error updating contact:', err);
    res.status(500).json({ error: 'Failed to update contact', details: err.message });
  }
});

// DELETE /api/contacts/:sessionId/:id - Delete a contact
router.delete('/:sessionId/:id', async (req, res) => {
  try {
    const { id } = req.params;

    // Get contact before deletion
    const contact = await getQuery(
      'SELECT * FROM contacts WHERE id = ? AND session_id = ?',
      [id, req.sessionId]
    );

    await runQuery(
      'DELETE FROM contacts WHERE id = ? AND session_id = ?',
      [id, req.sessionId]
    );

    // Trigger webhook for contact deleted
    try {
      if (contact) {
        await triggerWebhook(req.sessionId, 'contact.deleted', {
          contactId: id,
          phone: contact.phone,
          name: contact.name,
          timestamp: new Date()
        });
      }
    } catch (err) {
      console.error('Error triggering webhook:', err);
    }

    res.json({
      success: true,
      message: 'Contact deleted successfully'
    });
  } catch (err) {
    console.error('Error deleting contact:', err);
    res.status(500).json({ error: 'Failed to delete contact', details: err.message });
  }
});

module.exports = router;
