const express = require('express');
const router = express.Router();
const { v4: uuidv4 } = require('uuid');
const { runQuery, getQuery, allQuery } = require('../database');
const { validateApiKey } = require('../auth');

// Middleware to validate API key
async function validateApiKeyMiddleware(req, res, next) {
  const apiKey = req.headers['x-api-key'];
  const sessionId = req.params.sessionId;

  if (!apiKey || !sessionId) {
    return res.status(401).json({ error: 'Missing API key or session ID' });
  }

  const isValid = await validateApiKey(apiKey, sessionId);
  if (!isValid) {
    return res.status(401).json({ error: 'Invalid API key or session not authenticated' });
  }

  req.sessionId = sessionId;
  req.apiKey = apiKey;
  next();
}

// Apply middleware to all routes
router.use('/:sessionId', validateApiKeyMiddleware);

// POST /api/templates/:sessionId - Create a template
router.post('/:sessionId', async (req, res) => {
  try {
    const { name, content, variables } = req.body;

    if (!name || !content) {
      return res.status(400).json({ error: 'Name and content are required' });
    }

    const templateId = uuidv4();
    await runQuery(
      `INSERT INTO templates (id, session_id, name, content, variables)
       VALUES (?, ?, ?, ?, ?)`,
      [templateId, req.sessionId, name, content, JSON.stringify(variables || [])]
    );

    res.json({
      success: true,
      templateId: templateId,
      name: name,
      content: content
    });
  } catch (err) {
    console.error('Error creating template:', err);
    res.status(500).json({ error: 'Failed to create template', details: err.message });
  }
});

// GET /api/templates/:sessionId - Get all templates
router.get('/:sessionId', async (req, res) => {
  try {
    const templates = await allQuery(
      'SELECT * FROM templates WHERE session_id = ? ORDER BY name',
      [req.sessionId]
    );

    const formatted = templates.map(t => ({
      ...t,
      variables: JSON.parse(t.variables || '[]')
    }));

    res.json({
      success: true,
      total: templates.length,
      templates: formatted
    });
  } catch (err) {
    console.error('Error getting templates:', err);
    res.status(500).json({ error: 'Failed to get templates', details: err.message });
  }
});

// GET /api/templates/:sessionId/:id - Get a specific template
router.get('/:sessionId/:id', async (req, res) => {
  try {
    const template = await getQuery(
      'SELECT * FROM templates WHERE id = ? AND session_id = ?',
      [req.params.id, req.sessionId]
    );

    if (!template) {
      return res.status(404).json({ error: 'Template not found' });
    }

    template.variables = JSON.parse(template.variables || '[]');

    res.json({
      success: true,
      template: template
    });
  } catch (err) {
    console.error('Error getting template:', err);
    res.status(500).json({ error: 'Failed to get template', details: err.message });
  }
});

// PUT /api/templates/:sessionId/:id - Update a template
router.put('/:sessionId/:id', async (req, res) => {
  try {
    const { name, content, variables } = req.body;
    const { id } = req.params;

    await runQuery(
      `UPDATE templates SET name = ?, content = ?, variables = ? 
       WHERE id = ? AND session_id = ?`,
      [name, content, JSON.stringify(variables || []), id, req.sessionId]
    );

    res.json({
      success: true,
      message: 'Template updated successfully'
    });
  } catch (err) {
    console.error('Error updating template:', err);
    res.status(500).json({ error: 'Failed to update template', details: err.message });
  }
});

// POST /api/templates/:sessionId/:id/render - Render a template with variables
router.post('/:sessionId/:id/render', async (req, res) => {
  try {
    const { variables } = req.body;
    const { id } = req.params;

    const template = await getQuery(
      'SELECT * FROM templates WHERE id = ? AND session_id = ?',
      [id, req.sessionId]
    );

    if (!template) {
      return res.status(404).json({ error: 'Template not found' });
    }

    let content = template.content;
    
    // Replace variables in content
    if (variables) {
      Object.keys(variables).forEach(key => {
        content = content.replace(new RegExp(`{{${key}}}`, 'g'), variables[key]);
      });
    }

    res.json({
      success: true,
      rendered: content
    });
  } catch (err) {
    console.error('Error rendering template:', err);
    res.status(500).json({ error: 'Failed to render template', details: err.message });
  }
});

// DELETE /api/templates/:sessionId/:id - Delete a template
router.delete('/:sessionId/:id', async (req, res) => {
  try {
    const { id } = req.params;

    await runQuery(
      'DELETE FROM templates WHERE id = ? AND session_id = ?',
      [id, req.sessionId]
    );

    res.json({
      success: true,
      message: 'Template deleted successfully'
    });
  } catch (err) {
    console.error('Error deleting template:', err);
    res.status(500).json({ error: 'Failed to delete template', details: err.message });
  }
});

module.exports = router;
